// -------- High Resolution LPS Linux Library ------------
//
// RD 6/9/2017
// HME 06/13/2018   Started rewrite for libusb 1.0
// HME 08/19/2018   Restarted rewrite for libusb 1.0 - tagged as 0.99
// HME 08/20/2018   Removed conditional compile
// HME 08/24/2018   Added RD mutichannel support
// HME 12/22/2018   Corrected missing 602Q string - changed to 1.01 (skipping 1.00)
// HME 01/04/2019   Syntax fixes for C99 compatibility
// HME 03/29/2019   Made changes to the default ranges for the 602Q
// HME 04/03/2019   Added the 906V from Sean Scanlon's suggested changes
// RD  04/24/2019	Revised to fix FC19 segment fault in libusb_interrupt_transfer,
//					changing endpoints in HW, and device detection and open/close process
// RD  05/09/2019	This version has <libusb.h> to use the system installed version instead of a local copy
// HME 05/13/2019   Added in placeholders for missing functions
// RD  05/17/2019	Added functions
// RD  11/24/2019	Added multi-channel functions, along with support for new devices and long profiles
// RD  12/11/2019	Added timer based USB transaction throttle to avoid overrunning the devices on certain
//					host controllers.
// RD  12/12/2019	Reduced catnap times during reads to improve performance
// RD  8/16/2020	Added support for expandable LDA devices with up to 64 channels
// RD  9/21/2020	Revised the profile caching strategy, only the last used profile entry is cached now
// JA  1/30/2025    Added support for the 608, 802-2 and 303 devices
// NB  06/19/2025	Adapted for LPS devices
// NB  09/05/2025	Added support for LPS-202-8
// ----------------------------------------------------------------------------
// This library uses .05 degree units for phase angle values, so 10 degrees is represented by 200. (multiply by 20 to convert degrees to api units)

//#include "libusb.h"	// use a local copy, helpful on systems with legacy versions of libusb-1.0
#include <libusb-1.0/libusb.h>		// rely on the system installed copy
#include <linux/hid.h>	/* AK: Changed include for modern linux. */
#include <stdbool.h>	/* AK: Added include for 'bool' type */
#include <stdint.h>		// RD: used to avoid compiler cast warnings int -> void
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <time.h>
#include <errno.h>
#include "LPShid.h"

#define FALSE 0
#define TRUE 1			// RD -- aligned with ISO standard for bool type

#define PACKET_CTRL_LEN 8
#define PACKET_INT_LEN 8
#define INTERFACE 0
#define ENDPOINT_INT_IN1 0x81
#define ENDPOINT_INT_IN2 0x82
#define TIMEOUT 200   //500		RD 9/23/20 changed to 200 to make it shorter than the close thread timeout

#define LANGID 0x0409
#define USE_MISC 1  // HME: Makes Unistd.h happy

//#define LIBVER "0.98"	// RD:	added support for more devices, including High Res attenuators,
												//	internal representation of attenuation is now in .05 db units
//#define LIBVER "0.99" // HME: switched to libusb 1.0
//#define LIBVER "1.05" // Has extensive mods over 1.04
#define LIBVER "1.11"

#define LPS_LIBVERSION 0x10B		// we return an integer representation of the version with one byte each for major and minor version




void *brick_handler_function (void *ptr);
int SetChannel(DEVID deviceID, int channel);


/* ----------------------------------------------------------------- */
/* globals we'll be using at runtime */
char errmsg[32]; 			// For the status->string converter

bool bVerbose = FALSE; 		// True to generate debug oriented printf output during device enumeration
int Trace_Out = 0;          // variable to control how much tracing to do
int IO_Trace = 0;           // used for tracing control during device init and I/O operations

bool TestMode = TRUE; 		// if TestMode is true we fake it -- no HW access
							// TestMode defaults to FALSE for production builds

bool Did_libusb_init = FALSE;	// Flag used to prevent multiple calls to libusb_init, and to enable de-init when no devices are open

LPSPARAMS lps [MAXDEVICES]; // an array of structures each of which holds the info for a given
							// device. The DeviceID is the index into the array. Devices may come and go
							// so there is no guarantee that the active elements are contiguous
							// lps[0] is used as a temporary device info structure so we actually
							// only support MAXDEVICES-1 devices.

// we use timers for our read timeout, and to throttle the sending of USB commands
time_t starttime, currtime;
bool HasHRTimer = FALSE;
struct timespec mtimer_CTime, mtimer_LastSend;


// product names
const char sVNX1[] = "LPS-802";
const char sVNX2[] = "LPS-123";
const char sVNX3[] = "LPS-402";
const char sVNX4[] = "LPS-202";
const char sVNX5[] = "LPS-802-4";
const char sVNX6[] = "LPS-802-8";
const char sVNX7[] = "LPS-402-4";
const char sVNX8[] = "LPS-402-8";
const char sVNX9[] = "LPS-202-8";


// device VID and PID
const unsigned short devVID = 0x041f; // device VID for Vaunix Devices

const unsigned short LPS_Pids[34] = {
							0x0000,  // unused
							0x1240,  // device PID for Vaunix LPS-802
							0x1241,  // device PID for Vaunix LPS-123
							0x1242,  // device PID for Vaunix LPS-402
							0x1243,  // device PID for Vaunix LPS-202
							0x1244,  // device PID for Vaunix LPS-802-4
							0x1245,  // device PID for Vaunix LPS-802-8
							0x1246,  // device PID for Vaunix LPS-402-4
							0x1247,  // device PID for Vaunix LPS-402-8
							0x1249,  // device PID for Vaunix LPS-202-8
						};


/* stuff for the threads */
pthread_t threads[MAXDEVICES];

#define THREAD_IDLE 0
#define THREAD_START 1
#define THREAD_EXIT 3
#define THREAD_DEAD 4
#define THREAD_ERROR -1

// variables used to manage libusb 1.0 interaction
unsigned busnum = 0, devaddr = 0, _busnum, _devaddr;
libusb_device *dev, **devs;
libusb_device_handle *device = NULL;
struct libusb_device_descriptor desc;
libusb_context *LPS_ctx = NULL;


// to force LPS library debugging messages choose the following definitions
//#define DEBUG_OUT 0  	/* set this to 1 in order to see debugging output, 2 for a ton of output, or 3 for many tons */
//#define FORCE_LUSB_DEBUG 1

// otherwise use these definitions to allow application level control of the debug messages
#define DEBUG_OUT Trace_Out



/// ---------- Functions ----------------

/* usleep is deprecated, so this is a function using nanosleep() */
void catnap(long naptime) {
  // naptime comes in as the number of ms we want. 20 = 20ms
  struct timespec sleepytime;
   sleepytime.tv_sec = 0;
   sleepytime.tv_nsec = naptime * 1000000L;

   nanosleep(&sleepytime , NULL);
}

// --------------- Device IO support functions ----------------------------

bool CheckDeviceOpen(DEVID deviceID) {
  if (TestMode) return TRUE;// in test mode all devices are always available
  // Even for multichannel devices, we can use channel 0 to know if the device is open
  if ((lps[deviceID].DevStatus[0] & DEV_OPENED) && (deviceID != 0))
    return TRUE;
  else
    return FALSE;
}

// ------------------------------------------------------------------------
bool DevNotLocked(DEVID deviceID) {
  if (TestMode) return TRUE;// this shouldn't happen, but just in case...
  if (!(lps[deviceID].DevStatus[lps[deviceID].Channel] & DEV_LOCKED))
    return TRUE;// we return TRUE if the device is not locked!
  else
    return FALSE;
}

// ------------------------------------------------------------------------
void LockDev(DEVID deviceID, bool lock) {
  if (TestMode) return;// this shouldn't happen, but just in case...
  if (lock) {
    lps[deviceID].DevStatus[lps[deviceID].Channel] = lps[deviceID].DevStatus[lps[deviceID].Channel] | DEV_LOCKED;
    return;
  } else {
    lps[deviceID].DevStatus[lps[deviceID].Channel] = lps[deviceID].DevStatus[lps[deviceID].Channel] & ~DEV_LOCKED;
    return;
  }
}

// -- Clear the variables we cache to their unknown state --
void ClearDevCache(DEVID deviceID, int num_channels) {
	int ch;
	int i;

	if (num_channels > CHANNEL_MAX) num_channels = CHANNEL_MAX;	// protect our arrays

	for (ch = 0; ch < num_channels; ch++) {
		lps[deviceID].PhaseAngle[ch] = -1;			// we mark the phase angle as unknown to start
		lps[deviceID].WorkingFrequency[ch] = -1;	// and everybody else...
		lps[deviceID].RampStart[ch] = -1;
		lps[deviceID].RampStop[ch] = -1;
		lps[deviceID].PhaseStep[ch] = -1;
		lps[deviceID].PhaseStep2[ch] = -1;
		lps[deviceID].DwellTime[ch] = -1;
		lps[deviceID].DwellTime2[ch] = -1;
		lps[deviceID].IdleTime[ch] = -1;
		lps[deviceID].HoldTime[ch] = -1;
		lps[deviceID].Modebits[ch] = 0;				// Modebits aren't directly readable, but clearing them here for convenience
		lps[deviceID].ProfileIndex[ch] = -1;
		lps[deviceID].ProfileDwellTime[ch] = -1;
		lps[deviceID].ProfileIdleTime[ch] = -1;
		lps[deviceID].ProfileCount[ch] = -1;
		lps[deviceID].CachedProfileValue[ch] = -1;	// The int holds the index and the value, we init the whole thing to 0xFFFFFFFF to indicate it is empty.

		}
	}

/* A function to display the status as string */
char* fnLPS_perror(LVSTATUS status) {
  strcpy(errmsg, "STATUS_OK");
  if (BAD_PARAMETER == status) strcpy(errmsg, "BAD_PARAMETER");
  if (BAD_HID_IO == status) strcpy(errmsg, "BAD_HID_IO");
  if (DEVICE_NOT_READY == status) strcpy(errmsg, "DEVICE_NOT_READY");
  if (FEATURE_NOT_SUPPORTED == status) strcpy(errmsg, "FEATURE_NOT_SUPPORTED");
  if (DATA_UNAVAILABLE == status) strcpy(errmsg, "DATA_UNAVAILABLE");

  // Status returns for DevStatus
  if (INVALID_DEVID == status) strcpy(errmsg, "INVALID_DEVID");
  if (DEV_CONNECTED == status) strcpy(errmsg, "DEV_CONNECTED");
  if (DEV_OPENED == status) strcpy(errmsg, "DEV_OPENED");
  if (SWP_ACTIVE == status) strcpy(errmsg,  "SWP_ACTIVE");
  if (SWP_UP == status) strcpy(errmsg, "SWP_UP");
  if (SWP_REPEAT == status) strcpy(errmsg, "SWP_REPEAT");
  if (SWP_BIDIRECTIONAL == status) strcpy(errmsg, "SWP_BIDIRECTIONAL");
  if (PROFILE_ACTIVE == status) strcpy(errmsg, "PROFILE_ACTIVE");

  return errmsg;

}

bool CheckV2Features(DEVID deviceID) {
  // Even for multichannel devices, we can use channel 0 to know if the device is V2
  if (lps[deviceID].DevStatus[0] & DEV_V2FEATURES)
    {
      return TRUE;
    }
  else return FALSE;
}


bool CheckHiRes(DEVID deviceID) {
  // Even for multichannel devices, we can use channel 0 to know if the device is HiRes
  if (lps[deviceID].DevStatus[0] & DEV_HIRES)
    {
      return TRUE;
    }
  else return FALSE;
}

// functions based on hid_io.cpp
// RD 4-2019 modified to do the libusb interaction on the API thread instead of in the read thread and return better status
LVSTATUS VNXOpenDevice(DEVID deviceID) {

	int usb_status;
	int retries;
	int cnt, cntidx;
	libusb_device **devlist, *dev ;
	struct libusb_device_descriptor desc;
	int HWBusAddress = 0;


  if (!(lps[deviceID].DevStatus[0] & DEV_CONNECTED))	// we can't open a device that isn't there!
    return DEVICE_NOT_READY;

  if (lps[deviceID].DevStatus[0] & DEV_OPENED)			// the device is already open, BAD_PARAMETER gives a clue that it isn't just missing or disconnected
	return BAD_PARAMETER;

  if (!Did_libusb_init){
	usb_status = libusb_init(&LPS_ctx);		// we may need to re-init since we call libusb_exit when a user closes the last device
	if (usb_status != 0) return BAD_HID_IO;
	Did_libusb_init = TRUE;
  }

  // we'll open the device. First we have to locate it by using its physical location where we enumerated it

	cnt = libusb_get_device_list(LPS_ctx, &devlist);

    for (cntidx=0; cntidx<cnt; cntidx++) {
		dev = devlist[cntidx];
		if (DEBUG_OUT > 2) printf("Device Search: cntidx=%d cnt=%d dev=%p\r\n", cntidx, cnt, (void *)dev);


		usb_status = libusb_get_device_descriptor(dev, &desc);

		// gather the USB address for this device so we can find it in the future
		HWBusAddress = ((libusb_get_bus_number(dev) << 8) | (libusb_get_device_address(dev)));

		if (HWBusAddress == lps[deviceID].BusAddress) {

			// we found the device, lets open it so the user can interact with it
			if (DEBUG_OUT > 2) printf("Opening LPS device %04x:%04x at bus address %04x\r\n",
					desc.idVendor,
					desc.idProduct,
					HWBusAddress);

			usb_status = libusb_open(dev, &lps[deviceID].DevHandle);		// RD 4-2019 the device handle is now stored as a part of the LPS structure

			if ((DEBUG_OUT > 2) && (usb_status != 0)) printf("  libusb_open returned error status = %x\r\n", usb_status);

			if (usb_status) {

				// our device opened failed, not much we can do about that except clean up and let the caller know
				libusb_free_device_list(devlist, 1);
				return DEVICE_NOT_READY;
			}

			if (DEBUG_OUT > 1) printf("  Detaching kernel driver %04x:%04x\r\n", lps[deviceID].idVendor, lps[deviceID].idProduct);
			usb_status = libusb_detach_kernel_driver(lps[deviceID].DevHandle, 0);

			if ((DEBUG_OUT > 2) && (usb_status != 0)) printf("  Detaching kernel driver returned error status = %x\r\n", usb_status);

			// In theory the libusb kernel driver might already be attached, so we ignore the LIBUSB_ERROR_NOT_FOUND case
			if ((usb_status != 0) && (usb_status != LIBUSB_ERROR_NOT_FOUND)) {

			libusb_free_device_list(devlist, 1);
			// we might have a disconnected device, so check for that
				if (usb_status == LIBUSB_ERROR_NO_DEVICE) {
					lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] & ~DEV_CONNECTED;
					return DEVICE_NOT_READY;
				}

				return BAD_HID_IO;		// not necessarily IO layer, but something failed in the kernel stack below us
			}

			if (DEBUG_OUT > 2) printf("  Setting configuration %04x:%04x\r\n", lps[deviceID].idVendor, lps[deviceID].idProduct);
			usb_status = libusb_set_configuration (lps[deviceID].DevHandle, 1);

			if (DEBUG_OUT > 2) printf ("  successfully set configuration: %s\n", usb_status ? "failed" : "passed");

			if (usb_status != 0) {

				libusb_free_device_list(devlist, 1);
				// we might have a disconnected device, so check for that
				if (usb_status == LIBUSB_ERROR_NO_DEVICE) {
					lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] & ~DEV_CONNECTED;
					return DEVICE_NOT_READY;
				}

				return BAD_HID_IO;		// not necessarily IO layer, but something failed in the kernel stack below us
			}

			usb_status = libusb_claim_interface (lps[deviceID].DevHandle, 0);

			if (DEBUG_OUT > 1) printf ("claim interface: %s\n", usb_status ? "failed" : "passed");

			if (usb_status != 0) {

				libusb_free_device_list(devlist, 1);
				// we might have a disconnected device, so check for that
				if (usb_status == LIBUSB_ERROR_NO_DEVICE) {
					lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] & ~DEV_CONNECTED;
					return DEVICE_NOT_READY;
				}

				return BAD_HID_IO;		// not necessarily IO layer, but something failed in the kernel stack below us
			}

			// now we can start the read thread
			if (DEBUG_OUT > 2) printf("Starting thread %d\r\n", deviceID);
			lps[deviceID].thread_command = THREAD_START; /* open device and start processing */
			pthread_create(&threads[deviceID], NULL, brick_handler_function, (void*)(uintptr_t)deviceID);

			// wait here for the thread to start up and get out of the startup state
			retries = 10;
			while (retries && (lps[deviceID].thread_command == THREAD_START)) {
				catnap(100);  /* wait 100 ms */
				retries--;
			}

			if (DEBUG_OUT > 2) {
				printf("Read Thread Started, thread status = %d\r\n", lps[deviceID].thread_command);
			}

			// RSD Test
			catnap(100);

			if (lps[deviceID].thread_command == THREAD_START) {

				// for some reason our thread failed to startup and run, not much we can do but bail out
				libusb_free_device_list(devlist, 1);
				lps[deviceID].thread_command = THREAD_EXIT;

				// wait here for the thread to exit on its own
				retries = 10;
				while (retries && (lps[deviceID].thread_command != THREAD_DEAD)) {
					catnap(100);  /* wait 100 ms */
					retries--;
				}

				// the thread didn't exit on its own, so just cancel it
				if (lps[deviceID].thread_command != THREAD_DEAD) {
					pthread_cancel(threads[deviceID]);
				}

				return BAD_HID_IO;

			}

			// Even for multichannel devices, we use channel 0 to know if the device is open, so set the device open flag
			lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] | DEV_OPENED;

			// we only open one device, so we are done now
			libusb_free_device_list(devlist, 1);
			return STATUS_OK;

		}	// end of matching bus address case
	}
  // we failed to find a matching device if we get here
  libusb_free_device_list(devlist, 1);
  return DEVICE_NOT_READY;
}


// With the arrival of the expandable LPS devices it is easier to separate out the mask to channel function used for status reports
// from the general case used for responses
// Bit masks are used to identify the channel within either an 8 or 4 channel range, set by the width parameter
//

int BitMaskToChannel(int tid, int width, int mask)
{
	int channel;

	// We trap this case
	if (lps[tid].NumChannels == 1) return 0; // single channel devices only have channel 0.

	// the mask can be either 4 bits wide or 8 bits wide
	if (width == 4) {
		mask &= 0x0F;												  // clear out any stray bits for a 4 channel mask
	}
	else if (width == 8) {
		mask &= 0x0FF;											  // clear out any stray bits for an 8 channel mask
	}
	else {
		return 0;														  // we should never see this case
	}

	for (channel = 0; channel < width; channel++)
	{
		if (mask & 0x01) return channel;
		mask >>= 1;
	}

	return 0;								// hopefully we never see this failure case...
}


// -- Modified by RD 8/2020 to use the global channel setting when we have a device with more than 8 channels
//  for these devices we have to infer the source of the response and assume it is from our current channel
//
int MaskToChannel(int tid, int bank, int mask)
{
	int channel;

	if (lps[tid].NumChannels > CHANNEL_MAX) lps[tid].NumChannels = CHANNEL_MAX;		// just in case...

	if (lps[tid].NumChannels == 1) return 0;                 			// single channel devices only have channel 0.

	if (lps[tid].NumChannels > 8) {

		// if we have a logical bank number we use it. The caller will zero this argument when the command format does not allow for a logical bank number
		// in a response
		if (bank != 0) {
			channel = ((bank-1) * 8) + BitMaskToChannel(tid, 8, mask);	// combine the logical bank value with the channel encoded in the mask bits
		}
		else {
			// we don't have an explicit logical bank, so we will infer the channel as best we can
			if (lps[tid].Channel > 7) {
				channel = lps[tid].Channel;				// we have to rely on the current channel setting since the report does not have a full channel value
			}
			else {
				channel = BitMaskToChannel(tid, 8, mask);		// our channel is encoded in the mask, it is on the master
			}
		}
  }
	else {
		if (lps[tid].NumChannels == 4) {
			channel = BitMaskToChannel(tid, 4, mask);				// for 4 channel devices the active channel is encoded in the mask
		}
		else {
			channel = BitMaskToChannel(tid, 8, mask);				// for 8 channel devices the active channel is encoded in the mask
		}
	}

	return channel;
}

void decode_status(unsigned char swpdata, int tid, int ch) {

	lps[tid].DevStatus[ch] &= ~(SWP_ACTIVE | SWP_REPEAT | SWP_BIDIRECTIONAL | PROFILE_ACTIVE);	// preset all the zero bit cases

	// are we ramping?
	if (swpdata & (SWP_ONCE | SWP_CONTINUOUS))
		lps[tid].DevStatus[ch] = lps[tid].DevStatus[ch] | SWP_ACTIVE;

     // -- fill in the SWP_UP status bit
	if (swpdata & (SWP_DIRECTION))  // are we ramping downwards?
		lps[tid].DevStatus[ch] = lps[tid].DevStatus[ch] & ~SWP_UP;
	else
		lps[tid].DevStatus[ch] = lps[tid].DevStatus[ch] | SWP_UP;

	// -- fill in the continuous sweep bit
	if (swpdata & (SWP_CONTINUOUS))  // are we in continuous ramp mode?
		lps[tid].DevStatus[ch] = lps[tid].DevStatus[ch] | SWP_REPEAT;

	// -- fill in the bidirectional ramp bit
	if (swpdata & (SWP_BIDIR))  // are we in bi-directional ramp mode?
		lps[tid].DevStatus[ch] = lps[tid].DevStatus[ch] | SWP_BIDIRECTIONAL;

	// -- fill in the profile active bit
	if (swpdata & (STATUS_PROFILE_ACTIVE))  // is a phase profile playing?
		lps[tid].DevStatus[ch] = lps[tid].DevStatus[ch] | PROFILE_ACTIVE;

}

// -- RD modified to ensure command responses from multi-channel devices are stored in the correct channel
//	  support added for profile command related response reports
//
void report_data_decode(unsigned char rcvdata[], int tid) {
  int ch = 0;			// channel index, set to our single channel default
  int tmp;				// general purpose temp
  int i;
  unsigned short dataval_16;
  unsigned int dataval_32;
  char temp[32];


  if ((DEBUG_OUT > 2) && (rcvdata[0] != 0x4e)) {
    printf("Decoding ");
    for (i=0; i<8; i++)
      printf("%02x ", rcvdata[i]);
    printf(" active channel = %d\r\n", 1+lps[tid].Channel);  // The API channel is 1-org and lps[].Channel is 0-org
  }

  // pre-compute some common payload values for command responses
  dataval_16 = rcvdata[2] + (rcvdata[3]<<8);
  dataval_32 = rcvdata[2] + (rcvdata[3]<<8) + (rcvdata[4]<<16) + (rcvdata[5]<<24);

  if ((DEBUG_OUT > 2) && !((rcvdata[0] == VNX_STATUS) || (rcvdata[0] == VNX_HRSTATUS) || (rcvdata[0] == VNX_HR8STATUS)))
	  printf("Two byte data payload decodes to %d, four byte data payload to %d\r\n", dataval_16, dataval_32);


  // handle the status reports and responses from the device
  // multi-channel devices all include a channel mask with their response
  // for single channel devices we just use channel 0.


  switch(rcvdata[0]) {
  // handle the status report for a traditional single channel phase shifter
  case VNX_STATUS:
    if (DEBUG_OUT > 2) printf("VNX_STATUS: decoding\r\n");
    if (DevNotLocked(tid)) {
	  // update the phase angle, converting HW units to our .05 degree units
	  lps[tid].PhaseAngle[ch] = lps[tid].UnitScale * (rcvdata[5] + (rcvdata[6]<<8));

      if (DEBUG_OUT > 2) printf("  VNX_STATUS reports Phase Angle=%d\r\n", lps[tid].PhaseAngle[ch]);

	  // -- extract and save the status bits
	  decode_status(rcvdata[6], tid, ch);

      // -- fill in the current profile index --
	  lps[tid].ProfileIndex[ch] = rcvdata[4];
	  if (lps[tid].ProfileIndex[ch] > 99) lps[tid].ProfileIndex[ch] = 99;	// clip to profile length

      if (DEBUG_OUT > 2) printf("  VNX_STATUS sez Status=%02x\r\n", lps[tid].DevStatus[ch]);
      break;
    } /* if devnotlocked() */
    break;

  // handle the status report from a hi-res (and possibly 4 channel) phase shifter
	// RD 9/25/20 modified to handle the status report for a 4 channel phase shifter that is an expansion module
  case VNX_HRSTATUS:
    if (DEBUG_OUT > 2) printf("VNX_HRSTATUS: decoding\r\n");

    if (DevNotLocked(tid)) {
      if (lps[tid].NumChannels == 4 || lps[tid].Expandable) {
	    	// RD we don't want to change our global channel or channel mask based on a status report
	    	// but we need a temporary copy of the channel the HW is reporting
	    	// the channel mask bits are in the MSnibble of byteblock[1] instead of the LSnibble
	    	ch = BitMaskToChannel(tid, 4, (rcvdata[3] & 0xF0) >> 4);
				if (lps[tid].Expandable && rcvdata[1] != 0) {
					// RD we have a non-zero logical bank number for the channel, so adjust our channel accordingly
					ch = ch + 8*(rcvdata[1] & 0x07);
				}
      }

			// defend against a bad channel number
			if (ch >= CHANNEL_MAX) ch = CHANNEL_MAX -1;

      lps[tid].FrameNumber = (rcvdata[2] + (rcvdata[3]<<8)) & 0x000007FF;       // USB Frame Counter values are 11 bits

      lps[tid].PhaseAngle[ch] = (int)rcvdata[5] + 256 * (int)rcvdata[6];	// update the HiRes phase angle (in .05 degree units already)

      if (DEBUG_OUT > 2) printf("  VNX_HRSTATUS reports Phase Angle=%d for channel %d\r\n", lps[tid].PhaseAngle[ch], ch+1);

	  	// -- extract and save the status bits
	  	decode_status(rcvdata[6], tid, ch);

      // -- fill in the current profile index --
			tmp = rcvdata[1];
			tmp = (tmp << 4) & 0x00000300;	// clear out all other bits but b9 and b8
      lps[tid].ProfileIndex[ch] = tmp | rcvdata[4];
      if (lps[tid].ProfileIndex[ch] > lps[tid].ProfileMaxLength - 1) lps[tid].ProfileIndex[ch] = lps[tid].ProfileMaxLength - 1;	// clip to max profile length for this device

      if (DEBUG_OUT > 2) printf("  VNX_HRSTATUS sez Status=%02x\r\n", lps[tid].DevStatus[ch]);
      break;
    } /* if devnotlocked() */
    break;

// handle the status report from a hi-res 8 channel phase shifter and 8 channel expandable phase shifter modules
  case VNX_HR8STATUS:
    if (DEBUG_OUT > 2) printf("VNX_HR8STATUS: decoding\r\n");

    if (DevNotLocked(tid)) {
      if (lps[tid].NumChannels >= 8 || lps[tid].Expandable) {
	    	// RD we don't want to change our global channel or channel mask based on a status report
	    	//	  but we need a temporary copy of the channel the HW is reporting
	    	//    the channel mask bits are in the MSnibble of byteblock[1] instead of the LSnibble
				if (rcvdata[3] & 0x08){
					ch = BitMaskToChannel(tid, 8, (rcvdata[3] & 0xF0));	      // this status mask is for channels 4 to 7
				}
				else{
					ch = BitMaskToChannel(tid, 8, (rcvdata[3] & 0xF0) >> 4);  // this status mask is for channels 0 to 3
				}

				// for an expandable lps device we may have a status report from an expansion module
				if (lps[tid].Expandable && rcvdata[1] != 0) {
					ch = ch + 8*(rcvdata[1] & 0x07);
				}
      }

			// defend against a bad channel number
			if (ch >= CHANNEL_MAX) ch = CHANNEL_MAX -1;

      lps[tid].FrameNumber = (rcvdata[2] + (rcvdata[3]<<8)) & 0x000007FF; // USB Frame Counter values are 11 bits

      lps[tid].PhaseAngle[ch] = (int)rcvdata[5] + 256 * (int)rcvdata[6];	// update the HiRes phase angle level (in .05 degree units already)

      if (DEBUG_OUT > 2) printf("  VNX_HR8STATUS reports Phase Angle=%d\r\n", lps[tid].PhaseAngle[ch]);

	  	// -- extract and save the status bits
	  	decode_status(rcvdata[7], tid, ch);

      // -- fill in the current profile index --
			tmp = rcvdata[1];
			tmp = (tmp << 4) & 0x00000300;	// clear out all other bits but b9 and b8
      lps[tid].ProfileIndex[ch] = tmp | rcvdata[4];
      if (lps[tid].ProfileIndex[ch] > lps[tid].ProfileMaxLength - 1) lps[tid].ProfileIndex[ch] = lps[tid].ProfileMaxLength - 1;	// clip to max profile length for this device

      if (DEBUG_OUT > 2) printf("  VNX_HR8STATUS sez Channel %d Status=%02x\r\n", ch+1, lps[tid].DevStatus[ch]);
      break;
    } // if devnotlocked()
    break;

  // -- Decode the command responses --

  case VNX_FREQUENCY:
    if (DEBUG_OUT > 0) printf(" Working Frequency = %d\n", dataval_32);
    if (DevNotLocked(tid)){
      if(lps[tid].NumChannels > 1) {
		ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);		// responses from multi-channel devices include their channel ID, ch defaults to 0 for single channel devs
	  }
	  lps[tid].WorkingFrequency[ch] = dataval_32;
     }
    break;

  case VNX_PHASEANGLE:
    if (lps[tid].DevStatus[0] & DEV_HIRES) {
      dataval_16 = (int)rcvdata[2] + 256 * (int)rcvdata[3];	// HiRes data is already in .05 degree units
    }
    else {
      dataval_16 = lps[tid].UnitScale * (int)rcvdata[2];	// We have to scale the other devices HW units to our .05 degree units
    }
    if (DEBUG_OUT > 0) printf(" Phase Angle = %d\n", dataval_16);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
	  lps[tid].PhaseAngle[ch] = dataval_16;
	}
    break;

  case VNX_ADWELL:
    if (DEBUG_OUT > 0) printf(" Dwell Time = %d\n", dataval_32);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].DwellTime[ch] = dataval_32;
	}
    break;

  case VNX_ADWELL2:
    if (DEBUG_OUT > 0) printf(" 2nd Dwell Time = %d\n", dataval_32);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].DwellTime2[ch] = dataval_32;
	}
    break;

  case VNX_AIDLE:
    if (DEBUG_OUT > 0) printf(" Idle Time = %d\n", dataval_32);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].IdleTime[ch] = dataval_32;
		}
    break;

  case VNX_AHOLD:
    if (DEBUG_OUT > 0) printf(" Hold Time = %d\n", dataval_32);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].HoldTime[ch] = dataval_32;
		}
    break;

  case VNX_ASTART:
    if (lps[tid].DevStatus[0] & DEV_HIRES) {
      dataval_16 = (int)rcvdata[2] + 256 * (int)rcvdata[3];	// HiRes data is already in .05 degree units
    }
    else {
      dataval_16 = lps[tid].UnitScale * (int)rcvdata[2];	// We have to scale the other devices HW units to our .05 degree units
    }
    if (DEBUG_OUT > 0) printf(" Ramp Start Phase Angle = %d\n", dataval_16);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].RampStart[ch] = dataval_16;
		}
    break;

  case VNX_ASTEP:
    if (lps[tid].DevStatus[0] & DEV_HIRES) {
      dataval_16 = (int)rcvdata[2] + 256 * (int)rcvdata[3];	// HiRes data is already in .05 degree units
    }
    else {
      dataval_16 = lps[tid].UnitScale * (int)rcvdata[2];	// We have to scale the other devices HW units to our .05 degree units
    }
    if (DEBUG_OUT > 0) printf(" Phase Angle Step Size = %d\n", dataval_16);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].PhaseStep[ch] = dataval_16;
		}
    break;

  case VNX_ASTEP2:
    if (lps[tid].DevStatus[0] & DEV_HIRES) {
      dataval_16 = (int)rcvdata[2] + 256 * (int)rcvdata[3];	// HiRes data is already in .05 degree units
    }
    else {
      dataval_16 = lps[tid].UnitScale * (int)rcvdata[2];	// We have to scale the device's HW units to our .05 degree units
    }
    if (DEBUG_OUT > 0) printf(" Phase Angle Step Size Two = %d\n", dataval_16);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].PhaseStep2[ch] = dataval_16;
		}
    break;

  case VNX_ASTOP:
    if (lps[tid].DevStatus[0] & DEV_HIRES) {
      dataval_16 = (int)rcvdata[2] + 256 * (int)rcvdata[3];	// HiRes data is already in .05 degree units
    }
    else {
      dataval_16 = lps[tid].UnitScale * (int)rcvdata[2];	// We have to scale the device's HW units to our .05 degree units
    }
    if (DEBUG_OUT > 0) printf(" Ramp End Phase Angle = %d\n", dataval_16);
    if (DevNotLocked(tid)) {
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].RampStop[ch] = dataval_16;
		}
    break;

  case VNX_SWEEP:
    if (DEBUG_OUT > 0) printf(" Ramp Mode = %d\n", rcvdata[2]);
    if (DevNotLocked(tid)) {
 			// not generally used, same information available in the VNX_STATUS reports (and would be overwritten by the next status report...)
	  	if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	  }
	  	// -- extract and save the status bits
	  	decode_status(rcvdata[2], tid, ch);
    }
    break;

	// --- profile related responses ---
	case VNX_PROFILEDWELL:
	    if (DEBUG_OUT > 0) printf(" Profile Dwell Time = %d\n", dataval_32);
    if (DevNotLocked(tid)){
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].ProfileDwellTime[ch] = dataval_32;
	}
    break;

	case VNX_PROFILEIDLE:
	    if (DEBUG_OUT > 0) printf(" Profile Idle Time = %d\n", dataval_32);
    if (DevNotLocked(tid)){
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].ProfileIdleTime[ch] = dataval_32;
	}
    break;

	case VNX_PROFILECOUNT:
	    if (DEBUG_OUT > 0) printf(" Profile Count = %d\n", dataval_32);
    if (DevNotLocked(tid)){
		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }
      lps[tid].ProfileCount[ch] = dataval_32;
	}
    break;

  case VNX_SETPROFILE:
	if (DevNotLocked(tid)){

		if (lps[tid].DevStatus[0] & DEV_HIRES) {
			ch = MaskToChannel(tid, rcvdata[6], rcvdata[7]);
			i = rcvdata[5] & 0x03;			// devices with long profiles return the two high bits of the index in rcvdata[5]
			i = (i << 8) | rcvdata[4];		// HiRes devices report index in report[4]
			if ( i >= 0 && i < PROFILE_MAX_RAM){
				lps[tid].CachedProfileValue[ch] = ((i << 16) | (((int) dataval_16) & 0xFFFF));
			}
		}
		else {
			ch = MaskToChannel(tid, 0, rcvdata[7]);
			i = rcvdata[6];

			if ( i >= 0 && i < PROFILE_MAX){
				lps[tid].CachedProfileValue[ch] = lps[tid].UnitScale * (int)((rcvdata[2] + (rcvdata[3]<<8)) & 0xFFFF);
			}
		}

		if (DEBUG_OUT > 2) printf(" profile element cache = %x\n", lps[tid].CachedProfileValue[ch]);
		if (DEBUG_OUT > 1) printf(" profile element %d = %d\n", i, lps[tid].CachedProfileValue[ch] & 0xFFFF);
	}
	break;

  case VNX_CHANNEL:
	if (DevNotLocked(tid)){

		if (rcvdata[3] > 0) {
			lps[tid].GlobalChannel = (int) (rcvdata[3] - 1);	// our host side GlobalChannel is zero based, the packet uses a 1 based format

			if (lps[tid].Expandable == false) {
				lps[tid].Expandable = true;
			}

			lps[tid].Channel = lps[tid].GlobalChannel;
			if (lps[tid].Channel < 8) {
				lps[tid].ChMask = 0x01 << lps[tid].Channel;		// RD -- future proofing, the else case could handle all values of channel for now
			}
			else {
				lps[tid].ChMask = 0x01 << (lps[tid].Channel % 8);
			}
		}
		else {
			lps[tid].GlobalChannel = 0;		// this should not happen, but if we got a bad USB read it might
			break;												// bail out...
		}

		if (lps[tid].GlobalChannel > (CHANNEL_MAX-1)) {
			lps[tid].GlobalChannel = CHANNEL_MAX - 1;			// this should not happen, but we defend our arrays anyway
		}

		if(lps[tid].NumChannels > 1) {
		  ch = MaskToChannel(tid, 0, rcvdata[7]);	// responses from mc devs include the channel ID, ch defaults to 0 for single channel devs
	    }

		// get the number of expansion channels to find the total number of channels we have
		if (lps[tid].Expandable) {
			lps[tid].NumChannels = rcvdata[3] + lps[tid].BaseChannels;
		}

		if (DEBUG_OUT > 1) printf(" Global Channel %d of %d total channels\n", lps[tid].GlobalChannel, lps[tid].NumChannels);
		}

	break;

	// --- these are global parameters for a device, so not saved by channel ---
  case VNX_MINPHASE:
    if (lps[tid].DevStatus[0] & DEV_HIRES) {
      dataval_16 = (int)rcvdata[2] + 256 * (int)rcvdata[3];	// HiRes data is already in .05 degree units
    }
    else {
      dataval_16 = lps[tid].UnitScale * (int)rcvdata[2];		// We have to scale the other devices HW units to our .05 degree units
    }
    if (DEBUG_OUT > 0) printf(" Minimum Phase Angle = %d\n", dataval_16);
    if (DevNotLocked(tid))
      lps[tid].MinPhase = dataval_16;
    break;

  case VNX_MAXPHASE:
    if (lps[tid].DevStatus[0] & DEV_HIRES) {
      dataval_16 = (int)rcvdata[2] + 256 * (int)rcvdata[3];	// HiRes data is already in .05 degree units
    }
    else {
      dataval_16 = lps[tid].UnitScale * (int)rcvdata[2];		// We have to scale the other devices HW units to our .05 degree units
    }
    if (DEBUG_OUT > 0) printf(" Maximum Phase Angle = %d\n", dataval_16);
    if (DevNotLocked(tid))
      lps[tid].MaxPhase = dataval_16;
    break;

  case VNX_GETSERNUM:
    if (DEBUG_OUT > 0) printf(" Serial Number = %d\n", dataval_32);
    if (DevNotLocked(tid))
      lps[tid].SerialNumber = dataval_32;		// NB -- we only use this path during enumeration
    break;
  } /* switch */

  return;
}


// ************* The read thread handler for the brick ***********************
//
// RD 4-2019 modified to move open/close interaction with libusb to the
// Note -- 	the thread_command is used both for commands and thread status reporting
//			thread code must ensure that it has read the command before over-writing it, and must not overwrite a thread status
//
void *brick_handler_function (void *threadID) {
  int i, tid;
  tid = (int)(uintptr_t)threadID;
  int usb_status;
  char fullpath[128];
  int retries;
  int cnt, cntidx;
  int xfer_length;
  libusb_device **devlist, *dev ;
  libusb_device_handle *devhandle = NULL;

  if (DEBUG_OUT > 0) printf("Starting thread for device %d\r\n", tid);

  // -- top level thread loop --
  while ((lps[tid].thread_command >=0) && (lps[tid].thread_command != THREAD_EXIT)) {
    switch(lps[tid].thread_command) {
    case THREAD_IDLE:
      // this is where we launch a read and wait for incoming USB data
      usb_status = -1;
      retries = 10;
      while ((usb_status < 0) && (retries--) && (lps[tid].thread_command == THREAD_IDLE)) {

				if (DEBUG_OUT > 2) printf("Calling libusb_interrupt_transfer with DevHandle = %p, Endpoint = %x, rcvbuff = %p\r\n",(void *)lps[tid].DevHandle, lps[tid].Endpoint, (void *)lps[tid].rcvbuff);

				usb_status = libusb_interrupt_transfer(lps[tid].DevHandle,  // device handle
					lps[tid].Endpoint,	// endpoint
					lps[tid].rcvbuff,		// buffer
					PACKET_INT_LEN,			// max length
					&xfer_length,				// could be actual transfer length
					TIMEOUT);

				if (DEBUG_OUT > 2) printf("libusb_interrupt_transfer returned %x\r\n", usb_status);

				if (usb_status == LIBUSB_ERROR_NO_DEVICE) {
					// our device was unplugged, so we should clean up and end this thread
					lps[tid].DevStatus[0] = lps[tid].DevStatus[0] & ~DEV_CONNECTED;
					lps[tid].thread_command = THREAD_EXIT;
				}

				if (usb_status < 0 && (usb_status != LIBUSB_ERROR_TIMEOUT)) catnap(2); /* wait 2 ms before trying again for most errors */
      }

		  // did we get some data?
      if ((usb_status == 0) && (xfer_length > 0)) {
				if (DEBUG_OUT > 2) {
					printf("Thread %d reports %d...", tid, usb_status);
					for (i=0; i<usb_status; i++)
						printf("%02x ", lps[tid].rcvbuff[i]);
					printf("\r\n");
				}
				/* decode the HID data */
				report_data_decode(lps[tid].rcvbuff, tid);
				if (DEBUG_OUT > 2) printf("Decoded device %d data %02x, decodewatch=%02x\r\n", tid, lps[tid].rcvbuff[0], lps[tid].decodewatch);
				if (lps[tid].decodewatch == lps[tid].rcvbuff[0]) {
					if (DEBUG_OUT > 2) printf("Clearing decodewatch %02x for thread %d\r\n", lps[tid].decodewatch, tid);
					lps[tid].decodewatch = 0;
				}
      } else
				if (DEBUG_OUT > 0) perror("THREAD_IDLE");

      break;


    case THREAD_START:
		// this is the first code that the thread will run when started
		// since the device open was handled by our API level code we don't have much to do here


		// transition to the read state
		lps[tid].thread_command = THREAD_IDLE;
	  break;

    } /* switch */
  } /* while */

  // -- we received a THREAD_EXIT command
  if (DEBUG_OUT > 0) printf("Exiting thread for device %d because command=%d\r\n", tid, lps[tid].thread_command);
  lps[tid].thread_command = THREAD_DEAD;
  pthread_exit(NULL);
}

// -------------- SendReport -------------------------------------------------

bool SendReport(int deviceID, char command, char *pBuffer, int cbBuffer)
{
  int i;
  int send_status;
  int retries;
	int lastSendSeconds;
	long lastSendNanoSeconds;

  // Make sure the buffer that is being passed to us fits
  if (cbBuffer > HR_BLOCKSIZE) {
    // Report too big, don't send!
    return FALSE;
  }

  // lets make sure our command doesn't have any junk bits in it
  for (i=0; i<HID_REPORT_LENGTH; i++)
	  lps[deviceID].sndbuff[i] = 0;

  if (DEBUG_OUT > 1) printf("SR: command=%x cbBuffer=%d\r\n", (uint8_t)command, cbBuffer);
  lps[deviceID].sndbuff[0] = command;		// command to device
  lps[deviceID].sndbuff[1] = cbBuffer;
  for (i=0; i<cbBuffer; i++)
    lps[deviceID].sndbuff[i+2] = pBuffer[i];
  
  if (DEBUG_OUT > 1) {
    printf("SR to device %d: ", deviceID);
    for (i=0; i<8; i++) {
      printf("%02x ", (uint8_t)lps[deviceID].sndbuff[i]);
    }
    printf("\r\n");
  }

  // we wait for a file handle to appear in case we get here before the device really opened (unlikely...)
  retries = 0;
  while ((0 == lps[deviceID].DevHandle) && (retries++ < 10))
    sleep(1);

	// we may have to wait in order to ensure that very fast USB host controllers don't overrun our device
	if (HasHRTimer){
		clock_gettime(CLOCK_MONOTONIC_COARSE, &mtimer_CTime);
		lastSendSeconds = mtimer_CTime.tv_sec - mtimer_LastSend.tv_sec;
		lastSendNanoSeconds = mtimer_CTime.tv_nsec - mtimer_LastSend.tv_nsec;

		if (DEBUG_OUT > 2) printf("SR: last send was %d seconds and %ld nanoseconds ago\r\n", lastSendSeconds, lastSendNanoSeconds);

		if (lastSendSeconds == 1){
			// we may have rolled over, so check if our last time was at least 1 ms earlier
			lastSendNanoSeconds = mtimer_CTime.tv_nsec + (1000000000L - mtimer_LastSend.tv_nsec);
			if (lastSendNanoSeconds < 2000000L){																					// RD Testing, was 1000000L for 1ms minimum time
				// we want to wait enough to allow 2ms between USB Commands
				mtimer_CTime.tv_sec = 0;
				mtimer_CTime.tv_nsec = 2000000L - lastSendNanoSeconds;
				if (DEBUG_OUT > 2) printf("SR: rollover delay = %ld\r\n", mtimer_CTime.tv_nsec);
				nanosleep(&mtimer_CTime , NULL);
			}
		}
		else if (lastSendSeconds == 0 && lastSendNanoSeconds < 2000000L) {							// RD Testing, was 1000000L for 1ms minimum time
				// we want to wait enough to allow 2ms between USB Commands
				mtimer_CTime.tv_sec = 0;
				mtimer_CTime.tv_nsec = 2000000L - lastSendNanoSeconds;
				if (DEBUG_OUT > 2) printf("SR: delay = %ld\r\n", mtimer_CTime.tv_nsec);
				nanosleep(&mtimer_CTime , NULL);
		}
		else {
			// if lastSendSeconds > 1 or the nanoseconds portion is > 1 ms we don't have to worry about it...
			if (DEBUG_OUT > 2) printf("SR: no delay needed, last send was %d seconds and %ld nanoseconds ago\r\n", lastSendSeconds, lastSendNanoSeconds);
		}
	}
	else {
		// we don't have a timer, so we'll just wait 1ms
		 mtimer_CTime.tv_sec = 0;
		 mtimer_CTime.tv_nsec = 1000000L;
		 if (DEBUG_OUT > 2) printf("SR: no HR time fixed delay = %ld\r\n", mtimer_CTime.tv_nsec);
		 nanosleep(&mtimer_CTime , NULL);
	}

	// update the send time
	clock_gettime(CLOCK_MONOTONIC_COARSE, &mtimer_LastSend);

  /* we have data to write to the device */
  if (DEBUG_OUT > 1) printf("SR: sending the write to handle %p...\r\n", (void *)lps[deviceID].DevHandle);
  send_status = libusb_control_transfer(lps[deviceID].DevHandle,
				0x21,
				0x09, //HID_REPORT_SET,
				0x200,
				0,
				lps[deviceID].sndbuff,
				PACKET_CTRL_LEN,
				TIMEOUT);
  if (DEBUG_OUT > 1) printf("SR: sent it...\r\n");
  if (DEBUG_OUT > 1) {
    printf("(status=%d handle=%p)", send_status, (void *)lps[deviceID].DevHandle);
    if (send_status < 0) perror("SendReport"); else printf("\r\n");
  }
  return TRUE;
}


// ------------ GetParameter ---------------------------------------------
//
// The GetParam argument is the command byte sent to the device to get
// a particular value. The response is picked up by the read thread and
// parsed by it. The parser clears the corresponding event.


bool GetParameter(int deviceID, int GetParam)
{
	char VNX_param[6] = {0, 0, 0, 0, 0, 0};
	int timedout;

	if (DEBUG_OUT > 0) printf(" sending a GET command = %x\n", (char) GetParam );
	lps[deviceID].decodewatch = (char) GetParam;
	if (!SendReport(deviceID, (char)GetParam, VNX_param, 0)) {
	  return FALSE;
	}

	if (DEBUG_OUT > 0) printf(" SendReport sent a GET command successfully to device %d = %x\n", deviceID, (char) GetParam );

	starttime = time(NULL);
	timedout = 0;

	// wait until the value is decoded or 2 seconds have gone by
	// RD 4-2019 modified to yield the thread during the wait
	while ((lps[deviceID].decodewatch > 0) && (0 == timedout)) {

	  catnap(10);	// yield for 10 ms
	  if ((time(NULL)-starttime) > 2) timedout = 1;
	}

	return (0 == timedout) ? TRUE : FALSE;
}

// -------------- Get Routines to read device settings --------------------
//
// Note: for these functions deviceID is not checked for validity
//		 since it was already checked in the calling program.

bool GetPhaseAngle(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PHASEANGLE))
    return FALSE;

  return TRUE;
}

bool GetFrequency(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_FREQUENCY))
    return FALSE;

  return TRUE;
}

// -------------------------------

bool GetIdleTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_AIDLE))
    return FALSE;

  return TRUE;
}

// -------------------------------

bool GetHoldTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_AHOLD))
    return FALSE;

  return TRUE;
}
// -------------------------------

bool GetRampStart(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTART))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetRampStep(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTEP))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetRampStep2(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTEP2))
    return FALSE;

  return TRUE;
}
// -------------------------------

bool GetRampEnd(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTOP))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetDwellTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ADWELL))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetDwellTime2(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ADWELL2))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetWorkingFrequency(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_FREQUENCY))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetRampStop(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTOP))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetPhaseAngleStep(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTEP))
    return FALSE;

  return TRUE;
}

// -------------------------------
bool GetPhaseAngleStep2(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_ASTEP2))
    return FALSE;

  return TRUE;
}

// ---- profile related Get functions ------
bool GetProfileCount(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILECOUNT))
    return FALSE;

  return TRUE;
}

bool GetProfileDwellTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILEDWELL))
    return FALSE;

  return TRUE;
}

bool GetProfileIdleTime(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_PROFILEIDLE))
    return FALSE;

  return TRUE;
}

bool GetChannel(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_CHANNEL))
    return FALSE;

  return TRUE;
}


// RD -- Revised to use a single element cache, see the response parser for details
bool GetProfileElement(DEVID deviceID, int index)
{
    int max_index;
	unsigned int bank;
	unsigned int ch;
	int timedout;

	max_index = lps[deviceID].ProfileMaxLength - 1;

	if (max_index < 0 || max_index > PROFILE_MAX_RAM - 1) lps[deviceID].ProfileMaxLength = 50;	// this should never happen, if it does, pick the smallest profile size for safety

    if (index < 0 || index > max_index) return FALSE;    // bail out on a bad index...

	ch = lps[deviceID].Channel;
	if (ch >= CHANNEL_MAX) ch = 0;				// just defending our arrays, should never happen

	bank = (unsigned int)(ch / 8) + 1;		// the logical bank is incremented before it is transmitted
											// this lets us determine if we have a logical bank in the command packet


    BYTE VNX_command[] = {0, 0, 0, 0, 0, 0};

    if (CheckHiRes(deviceID)){
		VNX_command[2] = (BYTE)(index & 0x00FF);				// Index low byte
		VNX_command[3] = (BYTE)((index >> 8) & 0x03);			// Index high bits (only bits 0–1 used)
		VNX_command[4] = (BYTE)bank;							// Logical bank number
		VNX_command[5] = (BYTE)lps[deviceID].ChMask;			// Logical channel mask
	}
    else{
        VNX_command[4] = (BYTE)index;			// standard V2 LPS devices have the index in byteblock[4]
		VNX_command[5] = (BYTE)lps[deviceID].ChMask;			// Logical channel mask
	}
    // --- first we send the command out to the device --
		lps[deviceID].decodewatch = (char) VNX_SETPROFILE;
    if (!SendReport(deviceID, VNX_SETPROFILE | VNX_GET, VNX_command, 6))
        return FALSE;

    // --- then we wait for the read thread's parser to signal that it got the response
	starttime = time(NULL);
	timedout = 0;

	// wait until the value is decoded or 2 seconds have gone by
	// RD 4-2019 modified to yield the thread during the wait
	while ((lps[deviceID].decodewatch > 0) && (0 == timedout)) {
	  catnap(10);	// yield for 10 ms, speed matters here
	  if ((time(NULL)-starttime) > 2) timedout = 1;
	}
	return (0 == timedout) ? TRUE : FALSE;
}



bool GetSerNum(DEVID deviceID) {
  if (!GetParameter(deviceID, VNX_GETSERNUM))
    return FALSE;

  return TRUE;
}




/* functions to manage devices, not getting or retrieving data */
/*-------------------------------------------------------------*/

// function to get the serial number of the device, returns negative values for errors, or the actual serial number
// certain LDA-602Q devices return a different USB serial number string than their actual serial number
// This function returns the actual device serial number. While it is theoretically possible for a device to have a serial number of 0,
// this code assumes non-zero serial numbers.
//
// This function is called with an open devhandle and leaves it open for the caller to close
//
int GetSerialNumber(libusb_device_handle *devhandle, uint8_t sernum_string_desc_index, char* str_sernum, int * features)
{
	int i;
	int usb_status;
	int serial_number = 0;
	char rcvbuff[32];
	int TDev = 0;
	int retries;

	// quick sanity check
	if (devhandle == NULL) return DEVICE_NOT_READY;		// somehow we don't have a valid device handle, so bail out now

	// protect our receive buffer
	for (i=0; i<32; i++) rcvbuff[i]=0;

	if (features != NULL)
		*features = DEFAULT_FEATURES;

	// get the USB serial number string
	usb_status = libusb_get_string_descriptor_ascii(devhandle, sernum_string_desc_index, rcvbuff, sizeof(rcvbuff)-1);

	if (DEBUG_OUT > 1) printf("SerNum string %d = [%s]\r\n", sernum_string_desc_index, rcvbuff);
	if (usb_status < 0) return BAD_HID_IO;

	// the USB serial number string starts with SN: so we skip those characters
	serial_number = atoi(rcvbuff+3);

	// we can just use the USB serial number value
	if (str_sernum != NULL)
		strcpy(str_sernum, rcvbuff+3);

	return serial_number;

}

// This function finds the LPS devices and sets up the parameters for each device it finds
//
//
void FindVNXDevices()
{
  bool bFound;
  int hTemp;  				// temporary variable
  int HWType; 				// temporary variable for hardware/model type
  int HWMinPhase;			// temporary variable for default minimum phase shift
  int HWMaxPhase;			// temporary variable for default maximum phase shift
  int HWPhaseStep;			// temporary variable for the devices minimum phase step in .05 degree units
  int HWMinFrequency;		// temporary variable for HRes minimum frequency
  int HWMaxFrequency;		// temporary variable for HRes maximum frequency
  int HWUnitScale;			// temporary variable for unit scaling -- we use .05 degrees internally
  char HWName[MAX_MODELNAME];  		// temporary variable for the hardware model name
  char HWSerial[8]; 		// temporary holder for the serial number
  int HWSerialNumber;		// temporary holder for the device serial number
  int HWNumChannels;        // temporary holder for the number of channels
  bool HWExpandable;		// temporary holder for our flag to indicate a device which can have expansion channels
  int HWMaxProfileRAM;
  int HWMaxProfileSaved;
  bool HWHasHiRes;			// true for HiRes devices
  int HWFeatures;
  int HWBusAddress;			// temporary holder for the USB bus address of the device, used to find the device later
  int HWEndpoint;			// temporary holder for the HW endpoint used for status reports

  libusb_device_handle *devhandle = NULL;
  char sendbuff[8];
  char rcvbuff[32];
  int usb_status;

  int busnum;
  int busaddress;

  if (!Did_libusb_init){
	usb_status = libusb_init(&LPS_ctx);		// just in case...
	if (usb_status != 0) return;			// we can't really return an error, but hopefully we won't just crash
	Did_libusb_init = TRUE;
  }

#if FORCE_LUSB_DEBUG
//   libusb_set_debug(NULL, 0); 	/* RD - if we want lots of debug, let's see the libusb output too. 0 for none, 4 for a lot */
  libusb_set_option(NULL, LIBUSB_OPTION_LOG_LEVEL, 0);
#endif


#if 0
  // for libusb versions after 1.0.21 use set_option
  if (DEBUG_OUT > 2)
    libusb_set_option(NULL, LIBUSB_OPTION_LOG_LEVEL, 4); 	/* if we want lots of debug, let's see the USB output too. */
  else
    libusb_set_option(NULL, LIBUSB_OPTION_LOG_LEVEL, 0);
  //#else

  // for libusb versions before 1.0.22 use set_debug
  if (DEBUG_OUT > 2)
    libusb_set_debug(NULL, 4); 	/* if we want lots of debug, let's see the USB output too. */
  else
    libusb_set_debug(NULL, 0);
#endif

  libusb_device **devlist, *dev ;
  int r;
  ssize_t cnt;
  int c, i, a, j;
  int send_status, open_status;
  int cntidx;

  /* ... */

  // We need to remove devices from our table that are no longer connected ---
  // to do this we clear the "connected" flag for each table entry that is not open initially
  // then, as we find them we re-set the "connected" flag
  // anybody who doesn't have his "connected" flag set at the end is gone - we found it
  // previously but not this time

  for (i = 1; i<MAXDEVICES; i++){
    // even for multichannel devices we can look at channel 0 to see if it's open or connected
    if ((lps[i].SerialNumber != 0) && !(lps[i].DevStatus[0] & DEV_OPENED))
      lps[i].DevStatus[0] = lps[i].DevStatus[0] & ~DEV_CONNECTED;
  }

  // Grab the list of devices and see if we like any of them
  cnt = libusb_get_device_list(NULL, &devlist);
  for (cntidx=0; cntidx<cnt; cntidx++) {
    dev = devlist[cntidx];
    struct libusb_device_descriptor desc;

    usb_status = libusb_get_device_descriptor(dev, &desc);
    //if (DEBUG_OUT > 1) printf("Vendor: %04x PID: %04x (%d of %ld)\r\n", desc.idVendor, desc.idProduct, cntidx, cnt);

    HWType = 0;
    HWMinPhase = 0;
    HWMinFrequency = 0;
    HWMaxFrequency = 0;					// defaults for non HiRes devices
	HWHasHiRes = FALSE;
	HWMaxProfileRAM = PROFILE_MAX;
	HWMaxProfileSaved = PROFILE_MAX;	// preset the common case
    HWNumChannels = 1;  				// default for most devices
	HWExpandable = false;				// most hardware does not support adding expansion modules
	HWEndpoint = ENDPOINT_INT_IN2;		// default for many LPS devices

	// check to see if the device is one of our devices
	if (desc.idVendor == devVID) {
		// look through our PIDs to see if any match the device
		for (i = 1; i<(sizeof LPS_Pids / sizeof LPS_Pids[0]); i++) {
			if (LPS_Pids[i] == desc.idProduct){
				HWType = i;
				break;
			}
		}
	}

    if (DEBUG_OUT > 2) printf("Found HWType = %d\r\n", HWType);

    if (HWType) { /* we like this device and we'll keep it */

	  // gather the USB address for this device so we can find it in the future
	  busnum = (int)libusb_get_bus_number(dev);
	  busaddress = (int)libusb_get_device_address(dev);
	  if (DEBUG_OUT > 2) printf("USB bus = %d, address =%d\r\n", busnum, busaddress);

	  HWBusAddress = (busnum << 8) | busaddress;

	  if (DEBUG_OUT > 1) printf("Opening LPS device %04x:%04x type %d at bus address %02x\r\n",
				desc.idVendor,
				desc.idProduct,
				HWType, HWBusAddress);

	  // lets open the device to gather more information about it
      usb_status = libusb_open(dev, &devhandle);
      if (usb_status < 0) {
			if (DEBUG_OUT > 0) 	printf("FindVNXDevices sez: I can't open the device!\r\n");
			// The most likely cause of the libusb_open failure is a device we already have open for an application -- so skip over it
			continue;
      }
	  else {
		// --- our device open succeeded, lets set the parameters for each type of hardware ---
		switch(HWType) {
			case 1: // LPS-802
					strcpy(HWName, sVNX1);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units, but the HW works in 1 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 20;			// 1 degree units
					HWMinFrequency = 40000;		// 4000 MHz
					HWMaxFrequency = 80000;		// 8000 MHz
					break;

			case 2:	// LPS-123
					strcpy(HWName, sVNX2);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units, but the HW works in 1 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 20;			// 1 degree units
					HWMinFrequency = 80000;		// 8000 MHz
					HWMaxFrequency = 120000;	// 12000 MHz
					break;

			case 3:	// LPS-402
					strcpy(HWName, sVNX3);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units, but the HW works in 1 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 20;			// 1 degree units
					HWMinFrequency = 20000;		// 2000 MHz
					HWMaxFrequency = 40000;		// 4000 MHz
					break;

			case 4:  // LPS-202
					strcpy(HWName, sVNX4);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units, but the HW works in 1 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 20;			// 1 degree units
					HWMinFrequency = 10000;		// 1000 MHz
					HWMaxFrequency = 20000;		// 2000 MHz
					break;

			case 5:  // LPS-802-4
					strcpy(HWName, sVNX5);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 1;			// .05 degree units
					HWHasHiRes = TRUE;
					HWMaxProfileSaved = 25;
					HWMaxProfileRAM = 1000;

					HWMinFrequency = 40000;		// 4000 MHz
					HWExpandable = false;
					HWMaxFrequency = 80000;		// 8000 MHz
					HWNumChannels = 4;
					HWEndpoint = ENDPOINT_INT_IN1;
					break;

			case 6:  // LPS-802-8
					strcpy(HWName, sVNX6);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 1;			// .05 degree units
					HWHasHiRes = TRUE;
					HWMaxProfileSaved = 25;
					HWMaxProfileRAM = 1000;

					HWMinFrequency = 40000;		// 4000 MHz
					HWExpandable = false;
					HWMaxFrequency = 80000;		// 8000 MHz
					HWNumChannels = 8;
					HWEndpoint = ENDPOINT_INT_IN1;
					break;

			case 7:  // LPS-402-4
					strcpy(HWName, sVNX7);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 1;			// .05 degree units
					HWHasHiRes = TRUE;
					HWMaxProfileSaved = 25;
					HWMaxProfileRAM = 1000;

					HWMinFrequency = 20000;		// 2000 MHz
					HWExpandable = false;
					HWMaxFrequency = 40000;		// 4000 MHz
					HWNumChannels = 4;
					HWEndpoint = ENDPOINT_INT_IN1;
					break;

			case 8:  // LPS-402-8
					strcpy(HWName, sVNX8);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 1;			// .05 degree units
					HWHasHiRes = TRUE;
					HWMaxProfileSaved = 25;
					HWMaxProfileRAM = 1000;

					HWMinFrequency = 20000;		// 2000 MHz
					HWExpandable = false;
					HWMaxFrequency = 40000;		// 4000 MHz
					HWNumChannels = 8;
					HWEndpoint = ENDPOINT_INT_IN1;
					break;

			case 9:  // LPS-202-8
					strcpy(HWName, sVNX9);
					HWMaxPhase = 7200;			// 360 degrees, expressed in .05 degree units
					HWPhaseStep = 20;			// 1 degree steps
					HWUnitScale = 1;			// .05 degree units
					HWHasHiRes = TRUE;
					HWMaxProfileSaved = 25;
					HWMaxProfileRAM = 1000;

					HWMinFrequency = 10000;		// 1000 MHz
					HWExpandable = false;
					HWMaxFrequency = 20000;		// 2000 MHz
					HWNumChannels = 8;
					HWEndpoint = ENDPOINT_INT_IN1;
					break;

		} /* HWType switch */

		// Lets get the device's serial number so we can uniquely identify it
		// We can also use the USB BusNumber:Address to identify devices

		HWSerialNumber = GetSerialNumber(devhandle, desc.iSerialNumber, rcvbuff, &HWFeatures);

		// We should be done with the device now
		libusb_close(devhandle);

		if (HWSerialNumber < 0) continue;		// without a serial number we can't do anything with the device

		// find an open slot to save the data
		// lets see if we have this unit in our table of devices already
		// RD -- we use the device serial number in this phase, and USB bus:address during later opens to match up the device
		//
		bFound = FALSE;

		for (j = 1; j<MAXDEVICES; j++){
			if (lps[j].SerialNumber == HWSerialNumber) {
				// we already have the device in our table
				bFound = TRUE;
				// Even for multichannel devices, we can use channel 0 to see if we're connected
				lps[j].DevStatus[0] = lps[j].DevStatus[0] | DEV_CONNECTED; // its here, mark it as connected
				// at this point the device is present, but not in use, no sense looking more
				break;
			 }
			} // MAXDEVICES for loop

      } // open failed

      // if the device isn't in the table we need to add it
      if (!bFound) {
		hTemp = 0;
			for (i = 1; i<MAXDEVICES; i++) {
				if (lps[i].SerialNumber == 0) {
				hTemp = i;
				break;
				}
			} // end of for loop search for an empty slot in our array of devices

		/* save all of the data we've already acquired */
		if (hTemp) {
		  lps[hTemp].SerialNumber = HWSerialNumber;		            			// save the device's serial number
		  lps[hTemp].DevStatus[0] = lps[hTemp].DevStatus[0] | DEV_CONNECTED;	// mark it as present
		  strcpy (lps[hTemp].ModelName, HWName);     		    	    		// save the device's model name


		  if (lps[hTemp].SerialNumber == 0 || lps[hTemp].SerialNumber > 99999) {
			lps[hTemp].DevStatus[0] = lps[hTemp].DevStatus[0] | DEV_V2FEATURES;	// these devices have V2 firmware
		  }
		  else {
			lps[hTemp].DevStatus[0] = lps[hTemp].DevStatus[0] & ~DEV_V2FEATURES; // these don't
		  }

		  if (HWHasHiRes) { // Single channel HiRes, Quad and 8 channel HiRes lpss
				lps[hTemp].DevStatus[0] = lps[hTemp].DevStatus[0] | DEV_HIRES; 	// HiRes LPSs
			  }
		  else {
				lps[hTemp].DevStatus[0] = lps[hTemp].DevStatus[0] & ~DEV_HIRES;	// everybody else
		  }

		  // -- save our device type
		  lps[hTemp].DevType = HWType;

		  // -- set up the default channel number, always 0 for single channel devices --
		  if (DEBUG_OUT > 2) printf("Saving %d as the number of channels in the base LPS device\r\n", HWNumChannels);
		  lps[hTemp].BaseChannels = HWNumChannels;  // save the number of channels in the base device
		  lps[hTemp].NumChannels = HWNumChannels;	// for non expandable devices NumChannels = BaseChannels. For expandable devices we update NumChannels during the device init.
		  if (HWNumChannels == 4) lps[hTemp].DevStatus[0] = lps[hTemp].DevStatus[0] | HAS_4CHANNELS;
		  if (HWNumChannels == 8) lps[hTemp].DevStatus[0] = lps[hTemp].DevStatus[0] | HAS_8CHANNELS;
		  lps[hTemp].Expandable = HWExpandable;

		  // Note - the channel value gets updated later for multi-channel devices
		  //	    the number of channels for expandable devices also gets updated later
		  lps[hTemp].Channel = 0;
		  lps[hTemp].ChMask = 0x01;

		  // initialize our cache data structures for this device to the unknown state
		  ClearDevCache(hTemp, HWNumChannels);

		  // -- copy over the rest of the device parameters --
		  lps[hTemp].MinPhaseStep = HWPhaseStep;
		  lps[hTemp].MinPhase = HWMinPhase;
		  lps[hTemp].MaxPhase = HWMaxPhase;		    // default values for phase angle range
		  lps[hTemp].UnitScale = HWUnitScale;				    // size of hardware unit in .05 degree units
																// .05 degrees = 1, 1 degree = 20
		  lps[hTemp].MinFrequency = HWMinFrequency;
		  lps[hTemp].MaxFrequency = HWMaxFrequency;

		  lps[hTemp].ProfileMaxLength = HWMaxProfileRAM;
		  lps[hTemp].ProfileMaxSaved = HWMaxProfileSaved;

		  // The device has been closed so let's make sure we can find it again
		  // We use the BusAddress for opens by the user after detection
		  lps[hTemp].idVendor = desc.idVendor;
		  lps[hTemp].idProduct = desc.idProduct;
		  lps[hTemp].idType = HWType;
		  lps[hTemp].BusAddress = HWBusAddress;
		  lps[hTemp].Endpoint = HWEndpoint;
		  strcpy(lps[hTemp].Serialstr, rcvbuff);

		  if (DEBUG_OUT > 1) {
			printf("Stored as new device #%d\r\n", hTemp);
			printf("Serial number=%d\r\n", lps[hTemp].SerialNumber);
			printf("Devstatus=%08x\r\n", lps[hTemp].DevStatus[0]);
			printf("Model name=%s\r\n", lps[hTemp].ModelName);
			printf("MinPhase=%d\r\n", lps[hTemp].MinPhase);
			printf("MaxPhase=%d\r\n", lps[hTemp].MaxPhase);
			printf("Resolution=%d\r\n", lps[hTemp].MinPhaseStep);
			printf("Vendor ID=%04x\r\n", lps[hTemp].idVendor);
			printf("Product ID=%04x\r\n", lps[hTemp].idProduct);
			printf("Serial number=%s\r\n", lps[hTemp].Serialstr);
			printf("Number of channels=%d\r\n", lps[hTemp].NumChannels);
			printf("Profile Length=%d\r\n", lps[hTemp].ProfileMaxLength);
		  }
		}
		else {
			// our table of devices is full, not much we can do

		}
      } /* if !bfound  */
    } // if HWtype
  }	// end of for loop over the device list

  // -- we can free the list of devices now --
  //    this call may free individual device data objects that don't have an open handle to them, but it should not free
  //    the device structures for active (open) devices
  //
  libusb_free_device_list(devlist, 1);		// RD 4-2019 not devs, the devlist!

  /* clean up the structure and mark unused slots */
  for (i = 1; i<MAXDEVICES; i++){
    if ((lps[i].SerialNumber != 0) && !(lps[i].DevStatus[0] & DEV_CONNECTED))
      lps[i].SerialNumber = 0;	// mark this slot as unused

    if (lps[i].SerialNumber == 0)
      lps[i].DevStatus[0] = 0;		// clear the status for empty slots
  }	// end of zombie removal for loop

}

/* ----------------------------------------------------------------- */

void fnLPS_Init(void) {
  /* clear out the storage structure. Must be called once before anything else */
  int i,j;
  int status;
  int usb_status;

  for (i = 0; i<MAXDEVICES; i++){
	  ClearDevCache(i, CHANNEL_MAX);  // clear all cached variables to the unknown state

    for (j = 0; j<CHANNEL_MAX; j++) {
      lps[i].DevStatus[j] = 0;		// init to no devices connected, clear per channel status

	}
    lps[i].SerialNumber = 0;		// clear the serial number
    lps[i].ModelName[0] = 0;		// put a null string in each model name field
    lps[i].DevHandle = 0;			// clear the device handles
  }

  if (!Did_libusb_init){
	usb_status = libusb_init(&LPS_ctx);				// ensure we only call libusb_init once
	if (usb_status != 0){
		if (DEBUG_OUT > 0)  printf("libusb_init failed with usb_status = %d\r\n", usb_status);
		return;					// not easy to recover from this...
 }
	Did_libusb_init = TRUE;
  }

	// check what timer resources are available
	if (clock_getres(CLOCK_MONOTONIC_COARSE, &mtimer_CTime )){
		if (DEBUG_OUT > 0)  printf("Unable to use HR timer, error number %d\r\n", errno);
		HasHRTimer = FALSE;
	}
	else {
		HasHRTimer = TRUE;
		if (DEBUG_OUT > 0)  printf("HR timer resolution = %ld in nanoseconds\r\n", mtimer_CTime.tv_nsec);
		clock_gettime(CLOCK_MONOTONIC_COARSE, &mtimer_LastSend);
	}

  //  libusb_set_option(LPS_ctx, LIBUSB_OPTION_LOG_LEVEL, verbose);
}

void fnLPS_SetTestMode(bool testmode) {
  TestMode = testmode;
}

// We return an integer representation of the version with one byte each for major and minor version
int fnLPS_GetLibVersion()
{
	return LPS_LIBVERSION;
}


// Return the current device status, may be called before a device is opened (via fnLPS_InitDevice)
// The function returns the status for the currently selected channel, adding in the device wide status bits stored in channel 0 for multi-channel devices
// Note that only some status bits are valid for an unopened device
//
int fnLPS_GetDeviceStatus(DEVID deviceID) {
	int ChannelStatusMask = (PROFILE_ACTIVE | SWP_BIDIRECTIONAL | SWP_REPEAT | SWP_UP | SWP_ACTIVE);
	int ch = 0;

       if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}
	else
	{
		if (lps[deviceID].NumChannels > 1){
			ch = lps[deviceID].Channel;
			return (lps[deviceID].DevStatus[0] & ~ChannelStatusMask) | (lps[deviceID].DevStatus[ch] & ChannelStatusMask);
		}
		else
		  return lps[deviceID].DevStatus[0];
	}
}



//	The tracing features are intended to assist in debugging --
//	Note that DEBUG related #defines can alter the behavior of the debug printf code
//	IOTrace is used to control the debugging messages from the underlying libusb library
//	Review its documentation, and .h files for more detail
//	Note that libusb debug message behavior varies depending on the version of libusb being used.
//  If you want to use IO_Trace != 0 with libusb output, then call this function after fnLPS_Init

void fnLPS_SetTraceLevel(int trace, int devtrace, bool verbose)
    {
			const struct libusb_version * libversion = NULL;

			// --- enable libusb tracing using IO_Trace value---
      bVerbose = verbose;

      // --- general purpose tracing ---
      Trace_Out = trace;

      // --- device IO tracing ---
      IO_Trace = devtrace;

			if (bVerbose == TRUE ) {
	 			// enable libusb tracing
				libversion = libusb_get_version(); // RD!! The get_version function is not supported by FC19 libusb
				if (libversion != NULL) {

				if ((libversion->major == 1) && (libversion->micro >= 22)) {
					libusb_set_option(LPS_ctx, LIBUSB_OPTION_LOG_LEVEL, IO_Trace);
				}
				else {
					// libusb_set_debug(LPS_ctx, IO_Trace);
					// if (DEBUG_OUT > 1) printf("Returned from libusb_set_debug %d\r\n",IO_Trace);

				}
				if (DEBUG_OUT > 1) printf("Using Libusb version %d.%d.%d\r\n",libversion->major, libversion->minor, libversion->micro);
				}
			}
    }



int fnLPS_GetNumDevices() {
  int retval = 0;
  int NumDevices = 0;
  int i;

  // See how many devices we can find, or have found before
  if (TestMode){
    // construct a fake device

    lps[1].SerialNumber = 55102;
    lps[1].DevStatus[0] = lps[1].DevStatus[0] | DEV_CONNECTED | DEV_V2FEATURES;
    lps[1].idType = 1;
	lps[1].DevType = 1;				// The device type for an LPS-802
    lps[1].MinPhase = 0;			// 0 degrees is always the minimum
    lps[1].MaxPhase = 7200;			// 360 degrees in .05 degree units
    lps[1].UnitScale = 20;			// for consistency...
    lps[1].NumChannels = 1;			// most devices have 1 channel
	lps[1].ProfileMaxLength = 50;
	lps[1].MinPhaseStep = 20;
	lps[1].MinFrequency = 40000;	// 4000 MHz
	lps[1].MaxFrequency = 80000;	// 8000 MHz
	// Set some default starting values
	lps[1].PhaseAngle[0] = 0;			// Start with phase angle 0 degrees
	lps[1].WorkingFrequency[0] = 40000;	// Start with 4000 MHz
	lps[1].PhaseStep[0] = 20;		// 1 degree in .05 degree units
	lps[1].PhaseStep2[0] = 20;		// 1 degree in .05 degree units
	lps[1].RampStart[0] = 0;		// Start at 0 degrees
	lps[1].RampStop[0] = 7200;		// 360 degrees in .05 degree units
	lps[1].DwellTime[0] = 1000;		// 1 second dwell time
	lps[1].DwellTime2[0] = 1;		// 1 ms dwell time two
	lps[1].IdleTime[0] = 0;			// 0 seconds idle time
	lps[1].HoldTime[0] = 0;			// 0 seconds hold time
	lps[1].ProfileCount[0] = 50;	// 50 element profile
	lps[1].ProfileDwellTime[0] = 1;	// 1 ms profile dwell time
	lps[1].ProfileIdleTime[0] = 0; 	// 0 seconds profile idle time


    strcpy (lps[1].ModelName, "LPS-802");

    retval = 1;

  } else {
    // go look for some real hardware
    FindVNXDevices();

    // Total up the number of devices we have
    for (i = 1; i < MAXDEVICES; i++){
      if (lps[i].DevStatus[0] & DEV_CONNECTED) NumDevices++;
    }
    retval = NumDevices;

  }
  return retval;
}

int fnLPS_GetNumChannels(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].NumChannels;
}

int fnLPS_GetDevInfo(DEVID *ActiveDevices) {
  int i;
  int NumDevices = 0;

  if ( ActiveDevices == NULL) return 0; // bad array pointer, no place to put the DEVIDs

  for (i = 1; i < MAXDEVICES; i++){ // NB -- we never put an active device in lps[0] - so DEVIDs start at 1
    if (lps[i].DevStatus[0] & DEV_CONNECTED) {
      ActiveDevices[NumDevices] = i;
      NumDevices++;
    }
  }

  return NumDevices;
}

int fnLPS_GetModelName(DEVID deviceID, char *ModelName) {
  int NumChars = 0;

  if (deviceID >= MAXDEVICES){
    return INVALID_DEVID;
  }

  NumChars = strlen(lps[deviceID].ModelName);
  // If NULL result pointer, just return the number of chars in the name
  if ( ModelName == NULL) return NumChars;
  strcpy(ModelName, lps[deviceID].ModelName);

  return NumChars;
}

// InitDevice opens the device for use. To better support code that opens and closes the device a lot
// only a few key parameters are read during the open. The rest are read as needed.
// For expandable devices the number of channels is updated here, and the active channel is set to 0 for all multi-channel devices.

int fnLPS_InitDevice(DEVID deviceID) {
	LVSTATUS status;

  if ((deviceID >= MAXDEVICES) || (deviceID == 0)) {
    return INVALID_DEVID;
  }

  if (TestMode)
    lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] | DEV_OPENED;
  else {

    // Go ahead and open a handle to the hardware
    status = VNXOpenDevice(deviceID);

	if (status != STATUS_OK)  // VNXOpenDevice returns 0 (STATUS_OK) if the open succeeded, otherwise an error code
	  return status;

	// RD Test of sends before read thread has actually Started
	catnap(10);

	// if we have a multi channel device, we need to select a channel to start with
	// for now we set the device to channel 1 (0 internally)
	if (lps[deviceID].NumChannels > 1)
	{
		SetChannel(deviceID, 1);
	}

	// for expandable devices find out how many channels we have
	if (lps[deviceID].Expandable) {
		if (!GetChannel(deviceID)) {
			printf("Init get Channel error\r\n");
			return BAD_HID_IO;
		}
	}

    // Get the rest of the device parameters from the device
    if (DEBUG_OUT > 0) printf("Time to start getting parameters from device %d\r\n", deviceID);
    if (!GetPhaseAngle(deviceID)) {
	printf("Init get phase angle error\r\n");
	return BAD_HID_IO;
    }

    if (DEBUG_OUT > 1) printf("about to get Working Frequency\r\n");
    if (lps[deviceID].DevStatus[0] & DEV_HIRES) {		// for HiRes devices get the current working frequency
	 if (!GetFrequency(deviceID)) {
	   printf("Init get freq error\r\n");
	   return BAD_HID_IO;
	 }
    }

  } // end of real device open process case

  // if we got here everything worked OK
  return STATUS_OK;
}


int fnLPS_CloseDevice(DEVID deviceID) {

	int i;
    bool DevOpen = FALSE;
    int Trace_Save = 0;            // we save the global trace state and use Trace_Init for device init/de-init

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  if (TestMode)
    lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] & ~DEV_OPENED;
  else {

    // we use IO_Trace for the device init and close phases
    Trace_Save = Trace_Out;
    Trace_Out = IO_Trace;

    // Go ahead and close this hardware - the first step is to stop its read thread
    lps[deviceID].thread_command = THREAD_EXIT;

    // The thread handler will clear any pending reads on the device. We'll wait up to 1 second then give up.
		// RD 9/23/20 testing a longer wait
    int retries;
    retries = 100;
    while (retries && (lps[deviceID].thread_command != THREAD_DEAD)) {
      catnap(100);  /* wait 100 ms */
      retries--;
    }
    if (DEBUG_OUT > 2) printf("After telling the thread to close, we have thread_command=%d retries=%d\r\n", lps[deviceID].thread_command, retries);
    lps[deviceID].thread_command = THREAD_EXIT; // this command should never be read ...

	libusb_close(lps[deviceID].DevHandle);	// -- RD 4/2019 re-designed to handle libusb open/close interaction in the API threads instead of in the read thread

    // Mark it closed in our list of devices
    lps[deviceID].DevStatus[0] = lps[deviceID].DevStatus[0] & ~DEV_OPENED;
  }

   // if no devices remain open, then we should be able to call libusb's exit procedure.
  for (i = 1; i<MAXDEVICES; i++){
    if (lps[i].DevStatus[0] & DEV_OPENED) {
        DevOpen = TRUE;                             // we found an open LPS device, not time to clean up yet
        break;
     }
  }

  if (!DevOpen){
    // we don't have any open LPS devices, so we should be able to safely close libusb. We will re-open it on the next GetNumDevices, or device open
    if (Did_libusb_init){
		libusb_exit(LPS_ctx);		// close our libusb context
		Did_libusb_init = FALSE;
	}
  }

  Trace_Out = Trace_Save;            // restore the normal trace level
  return STATUS_OK;

}

int fnLPS_GetSerialNumber(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].SerialNumber;
}


// Functions to set parameters

// Set the phase angle - the function uses 1 degree units
LVSTATUS fnLPS_SetPhaseAngle(DEVID deviceID, int phase) {

	int phase_20 = phase * 20;	// convert to .05 degree units
	int tmp_phase = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	LockDev(deviceID, TRUE);

	int old_phase = lps[deviceID].PhaseAngle[lps[deviceID].Channel];

	if ((phase_20 >= lps[deviceID].MinPhase) && (phase_20 <= lps[deviceID].MaxPhase)) {
	  lps[deviceID].PhaseAngle[lps[deviceID].Channel] = phase_20;
	  if (TestMode){
	    LockDev(deviceID, FALSE);
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	  }
	} else {
	  LockDev(deviceID, FALSE);
	  return BAD_PARAMETER;
	}

	// the phase angle value is OK, lets send it to the hardware

	if (lps[deviceID].UnitScale > 0)
		tmp_phase = lps[deviceID].PhaseAngle[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_phase;

	if (!SendReport(deviceID, VNX_PHASEANGLE | VNX_SET, ptr, 4)) {
	  lps[deviceID].PhaseAngle[lps[deviceID].Channel] = old_phase;
	  LockDev(deviceID, FALSE);
	  return BAD_HID_IO;
	}

	LockDev(deviceID, FALSE);

	return STATUS_OK;
}


// Set the phase angle for a quad (or other multi-channel) phase shifter - the function uses 1 degree units
LVSTATUS fnLPS_SetPhaseAngleQ(DEVID deviceID, int phase, int channel) {

	LVSTATUS SCResult;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	// set the channel first
	SCResult = (LVSTATUS) SetChannel(deviceID, channel);
		if ( SCResult != STATUS_OK) return SCResult;

	return fnLPS_SetPhaseAngle(deviceID, phase);
}

LVSTATUS fnLPS_SetPhaseAngleMC(DEVID deviceID, int phase, unsigned long long channelmask) {
	unsigned int i;
	int ScaledPhaseAngle = 0;
	int cmdlen;
	unsigned long long tmp64;
	unsigned int chmask;				// the channel mask for the current logical bank
	unsigned int logicalbankid;			// the current logical bank id, starting at 1 and increasing for each bank we have
	unsigned int num_banks;				// used to loop over the number of active banks we have in out mask
	unsigned int ch;					// our internal channel index used for updating caches
	unsigned int tmask;					// a test mask
	int old_phaseangle[64];				// somewhere to put the old phase angle values in case we hvae to back out due to an IO error
	BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

	phase = phase * 20;	// convert phase angle to .05 degree units

	if (deviceID >= MAXDEVICES)
	{
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID))
	{
		return DEVICE_NOT_READY;
	}

	// check to see if the phase angle value is reasonable
	if ((phase < lps[deviceID].MinPhase) || (phase > lps[deviceID].MaxPhase))
	{
		return BAD_PARAMETER;
	}

	// generate our scaled value for this hardware
	if (lps[deviceID].UnitScale > 0)
		ScaledPhaseAngle = phase / lps[deviceID].UnitScale;

	// build the parts of the command packet that don't change across banks
	VNX_command[0] = (BYTE)(ScaledPhaseAngle & 0x000000ff);			// low byte of phase angle in .05 degree device units
	VNX_command[1] = (BYTE)((ScaledPhaseAngle & 0x0000ff00) >> 8);	// high byte of phase angle

	// our commands with logical bank and channel addresses have a 6 byte payload
	// but in case somebody runs this function with an old single channel device, I'll generate that command length format too
	if (CheckHiRes(deviceID))
	{
		cmdlen = 6;				// command length is set to 6 now, although some instances of this command being sent with length = 4 probably exist
		// the firmware doesn't care
	}
	else
	{
		cmdlen = 1;				// this was 4 in earlier versions, should be 1, firmware doesn't care.
	}

	// test our channel mask and clip it to our device's maximum number of channels
	// we don't return an error since applications level code might just use "ALL" as 
	// a quick way of setting every channel
	tmp64 = 0xffffffffffffffff >> (64 - lps[deviceID].NumChannels);
	channelmask &= tmp64;

	// get rid of an easy error case, nothing to do if we don't have any channel bits set
	// but let the user know their channel mask parameter was invalid for the device
	if (channelmask == 0)
	{
		return BAD_PARAMETER;
	}

	// determine how many logical banks our device has
	num_banks = lps[deviceID].NumChannels / 8;
	if ((lps[deviceID].NumChannels % 8) != 0)
	{
		num_banks++;
	}

	// loop over all of our logical banks, sending down the multiple channel set command for that bank and
	// saving the corresponding attenuation values for each channel

	LockDev(deviceID, TRUE);

	for (i = 0; i < num_banks; i++)
	{
		// select the 8 bit wide mask for the bank
		chmask = (channelmask >> (i * 8)) & 0xFF;

		// if we find an empty bank just skip over it
		if (!chmask) continue;

		// save a copy of the existing attenuation values for this logical bank's active channels and
		// update the cached attenuation value
		tmask = 1;
		for (ch = i * 8; ch < (i * 8 + 8); ch++)
		{
			if (chmask & tmask)
			{
				old_phaseangle[ch] = lps[deviceID].PhaseAngle[ch];
				lps[deviceID].PhaseAngle[ch] = phase;
			}
			tmask <<= 1;
		}

		// if we are in test mode we don't need to update the actual hardware
		if (TestMode) continue;

		// time to send the command for this logical bank to the device
		if (Trace_Out > 2) printf("SetAttenMC: DeviceID = %d, Internal BankID = %d, chmask = %d. Attenuation = %d\n", deviceID, i, chmask, ScaledPhaseAngle);

		VNX_command[4] = (BYTE)(i + 1);		// the logical bank ID is sent as a 1 to N value
		VNX_command[5] = (BYTE)chmask;		// the logical bank's active channel mask

		if (!SendReport(deviceID, VNX_PHASEANGLE | VNX_SET, VNX_command, 6))
		{
			// something failed in the I/O process so we will try to back out as best we can and bail out
			tmask = 1;
			for (ch = i * 8; ch < (i * 8 + 8); ch++)
			{
				if (chmask & tmask)
				{
					lps[deviceID].PhaseAngle[ch] = old_phaseangle[ch];
				}
				tmask <<= 1;
			}

			LockDev(deviceID, FALSE);
			return BAD_HID_IO;
	  }

		// go back and do the next logical bank
  }

  // it looks like we successfully sent all of the bank specific commands so return success 
  LockDev(deviceID, FALSE);
  return STATUS_OK;
}

// Set the phase angle ramp start value
LVSTATUS fnLPS_SetRampStart(DEVID deviceID, int rampstart) {
	int rampstart_20 = rampstart * 20;	// convert to .05 degree units
	int tmp_rampstart = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_rampstart = lps[deviceID].RampStart[lps[deviceID].Channel];

	if ((rampstart_20 >= lps[deviceID].MinPhase) && (rampstart_20 <= lps[deviceID].MaxPhase)) {
	  lps[deviceID].RampStart[lps[deviceID].Channel] = rampstart_20;
	  if (TestMode){
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	  }
	} else {
	  return BAD_PARAMETER;
	}

	// the ramp start value is OK, lets send it to the hardware
	if (lps[deviceID].UnitScale > 0)
		tmp_rampstart = lps[deviceID].RampStart[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_rampstart;

	if (!SendReport(deviceID, VNX_ASTART | VNX_SET, ptr, 4)){
	  lps[deviceID].RampStart[lps[deviceID].Channel] = old_rampstart;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the phase angle ramp end value
LVSTATUS fnLPS_SetRampEnd(DEVID deviceID, int rampstop) {
	int rampstop_20 = rampstop * 20;	// convert to .05 degree units
	int tmp_rampstop = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_rampstop = lps[deviceID].RampStop[lps[deviceID].Channel];

	if ((rampstop_20 >= lps[deviceID].MinPhase) && (rampstop_20 <= lps[deviceID].MaxPhase)){

	  lps[deviceID].RampStop[lps[deviceID].Channel] = rampstop_20;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}
	// the ramp end value is OK, lets send it to the hardware
	if (lps[deviceID].UnitScale > 0)
	  tmp_rampstop = lps[deviceID].RampStop[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_rampstop;

	if (!SendReport(deviceID, VNX_ASTOP | VNX_SET, ptr, 4)) {
	  lps[deviceID].RampStop[lps[deviceID].Channel] = old_rampstop;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the phase angle ramp step size for the first phase
LVSTATUS fnLPS_SetPhaseAngleStep(DEVID deviceID, int phasestep) {
	int phasestep_20 = phasestep * 20;	// convert to .05 degree units
	int tmp_phasestep = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_phasestep = lps[deviceID].PhaseStep[lps[deviceID].Channel];

	if (phasestep_20 < (lps[deviceID].MaxPhase - lps[deviceID].MinPhase)) {
	  lps[deviceID].PhaseStep[lps[deviceID].Channel] = phasestep_20;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}
	// the phase ramp step value is OK, lets send it to the hardware
	if (lps[deviceID].UnitScale > 0)
		tmp_phasestep = lps[deviceID].PhaseStep[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_phasestep;

	if (!SendReport(deviceID, VNX_ASTEP | VNX_SET, ptr, 4)) {
		lps[deviceID].PhaseStep[lps[deviceID].Channel] = old_phasestep;
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the phase angle ramp step size for the second phase of the ramp
//
LVSTATUS fnLPS_SetPhaseAngleStepTwo(DEVID deviceID, int phasestep2) {
	int phasestep2_20 = phasestep2 * 20;	// convert to .05 degree units
	int tmp_phasestep2 = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	int old_phasestep2 = lps[deviceID].PhaseStep2[lps[deviceID].Channel];

	if (phasestep2_20 < (lps[deviceID].MaxPhase - lps[deviceID].MinPhase)) {
	  lps[deviceID].PhaseStep2[lps[deviceID].Channel] = phasestep2_20;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}
	// the phase ramp step value is OK, lets send it to the hardware
	if (lps[deviceID].UnitScale > 0)
		tmp_phasestep2 = lps[deviceID].PhaseStep2[lps[deviceID].Channel] / lps[deviceID].UnitScale;

	unsigned char *ptr = (unsigned char *) &tmp_phasestep2;

	if (!SendReport(deviceID, VNX_ASTEP2 | VNX_SET, ptr, 4)) {
		lps[deviceID].PhaseStep2[lps[deviceID].Channel] = old_phasestep2;
		return BAD_HID_IO;
	}
	return STATUS_OK;
}

// Functions to set parameters
// Channel is 1 to 4 or 1 to 8, or 1 to N for expandable devices at the API level

int SetChannel(DEVID deviceID, int channel)
{
	char VNX_command[] = { 0, 0, 0, 0, 0, 0 };

    if (deviceID >= MAXDEVICES){
    return INVALID_DEVID;
  }

  if (!CheckDeviceOpen(deviceID)){
    return DEVICE_NOT_READY;
  }

  if (channel < 1 || channel > lps[deviceID].NumChannels){
    return BAD_PARAMETER;
  }
	// we use zero based channel numbers internally, the API is 1 based, as is the new format command
	lps[deviceID].Channel = channel - 1;
	lps[deviceID].GlobalChannel = channel - 1;

  if (lps[deviceID].Expandable && (channel > (lps[deviceID].BaseChannels))) {
		// the channel is located on an expansion module so we use the new format command
		VNX_command[0] = (char)channel;

		lps[deviceID].ChMask = 0x01 << (channel % 8);	// This value is most likely unused for a matrix device, but it represents the channel mask for the active bank
														// RD!!!! looks like it should be channel - 1 %8

  }
  else {
	// convert to our mask form and save the mask
	lps[deviceID].ChMask = 0x01 << lps[deviceID].Channel;
    VNX_command[5] = (char)lps[deviceID].ChMask;

  }
  if (TestMode) {
	return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
  }
    // -- send the channel selection to the phase shifter --
    if (!SendReport(deviceID, VNX_CHANNEL | VNX_SET, VNX_command, 6))
      return BAD_HID_IO;

  return STATUS_OK;
}

LVSTATUS fnLPS_SetChannel(DEVID deviceID, int channel)
{
	LVSTATUS result;
	result = SetChannel(deviceID, channel);
	if(!result)
	{
		usleep(700000);  // 700msec
	}
	
	return result;
}


// Set the working frequency for HiRes LPS devices, frequency is in 100KHz units
LVSTATUS fnLPS_SetWorkingFrequency(DEVID deviceID, int frequency) {
	int tmp_frequency = 0;
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_frequency = lps[deviceID].WorkingFrequency[ch];

	if (DEBUG_OUT > 0) {
	  printf("Min frequency is %d\r\n", lps[deviceID].MinFrequency);
	  printf("Max frequency is %d\r\n", lps[deviceID].MaxFrequency);
	  printf("User frequency to set is %d\r\n", frequency);
	}

	if ((frequency >= lps[deviceID].MinFrequency) && (frequency <= lps[deviceID].MaxFrequency)) {
	  lps[deviceID].WorkingFrequency[ch] = frequency;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the working frequency value is OK, lets send it to the hardware
	tmp_frequency = frequency;
	unsigned char *ptr = (unsigned char *) &tmp_frequency;

	if (DEBUG_OUT > 0) printf("setting frequency to %d\r\n", tmp_frequency);

	if (!SendReport(deviceID, VNX_FREQUENCY | VNX_SET, ptr, 4)) {
	  lps[deviceID].WorkingFrequency[ch] = old_frequency;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the time to dwell at each step on the first phase of the ramp
LVSTATUS fnLPS_SetDwellTime(DEVID deviceID, int dwelltime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_dwelltime = lps[deviceID].DwellTime[ch];

	if (dwelltime >= VNX_MIN_DWELLTIME) {
	  lps[deviceID].DwellTime[ch] = dwelltime;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].DwellTime[ch];

	if (!SendReport(deviceID, VNX_ADWELL | VNX_SET, ptr, 4)) {
	  lps[deviceID].DwellTime[ch] = old_dwelltime;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the time to dwell at each step on the second phase of the ramp
LVSTATUS fnLPS_SetDwellTimeTwo(DEVID deviceID, int dwelltime2) {
	int ch = 0;
	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_dwelltime2 = lps[deviceID].DwellTime2[ch];

	if (dwelltime2 >= VNX_MIN_DWELLTIME) {
	  lps[deviceID].DwellTime2[ch] = dwelltime2;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].DwellTime2[ch];

	if (!SendReport(deviceID, VNX_ADWELL2 | VNX_SET, ptr, 4)) {
	  lps[deviceID].DwellTime2[ch] = old_dwelltime2;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}


// Set the time to wait at the end of the ramp
LVSTATUS fnLPS_SetIdleTime(DEVID deviceID, int idletime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_idletime = lps[deviceID].IdleTime[ch];

	if (idletime >= VNX_MIN_DWELLTIME) {
	  lps[deviceID].IdleTime[ch] = idletime;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the idle time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].IdleTime[ch];

	if (!SendReport(deviceID, VNX_AIDLE | VNX_SET, ptr, 4)) {
	  lps[deviceID].IdleTime[ch] = old_idletime;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the time to wait between the first and second phase of a bidirectional ramp
LVSTATUS fnLPS_SetHoldTime(DEVID deviceID, int holdtime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	int old_holdtime = lps[deviceID].HoldTime[ch];

	if (holdtime >= 0) {		// holdtime can be zero
	  lps[deviceID].HoldTime[ch] = holdtime;
	  if (TestMode)
	    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	} else {
	  return BAD_PARAMETER;
	}

	// the hold time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].HoldTime[ch];

	if (!SendReport(deviceID, VNX_AHOLD | VNX_SET, ptr, 4)) {
	  lps[deviceID].HoldTime[ch] = old_holdtime;
	  return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the ramp direction -- "up" is TRUE to ramp upwards
LVSTATUS fnLPS_SetRampDirection(DEVID deviceID, bool up) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	if (up)
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] & ~SWP_DIRECTION;	// ramp or sweep direction up (bit == 0)
	else
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] | SWP_DIRECTION;	// ramp or sweep direction downwards


	return STATUS_OK;
}

// Set the ramp mode -- mode = TRUE for repeated ramp, FALSE for one time ramp
LVSTATUS fnLPS_SetRampMode(DEVID deviceID, bool mode) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	if (mode) {
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] | SWP_CONTINUOUS;		// Repeated ramp or sweep
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] & ~SWP_ONCE;
	} else {
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] | SWP_ONCE;			// one time ramp or sweep
	  lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] & ~SWP_CONTINUOUS;
	}

	return STATUS_OK;
}

// Select bidirectional or unidirectional ramp mode
// TRUE for bidirectional ramp, FALSE for unidirectional ramp
LVSTATUS fnLPS_SetRampBidirectional(DEVID deviceID, bool bidir_enable) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

	if (bidir_enable)
	{
		lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] | SWP_BIDIR;	// bidirectional ramp
	}
	else
	{
		lps[deviceID].Modebits[ch] = lps[deviceID].Modebits[ch] & ~SWP_BIDIR;	// unidirectional ramp
	}

	return STATUS_OK;
}



// Start the ramp
LVSTATUS fnLPS_StartRamp(DEVID deviceID, bool go) {
	int icount;
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

	char VNX_ramp[] = {0, 0, 0, 0, 0, 0};

if (go)
	{
		icount = 3;														// the new phase shifters use 3 bytes
		VNX_ramp[0] = (char) lps[deviceID].Modebits[ch] & MODE_SWEEP;	// mode sweep for V2 includes the 0x10 bit
	}
	else
	{
		icount = 3;
		VNX_ramp[0] = 0;
	}

	if (TestMode)
	  return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW

	if (DEBUG_OUT > 0) printf(" sending a ramp command = %x\n", VNX_ramp[0] );

	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_ramp, icount))
	  return BAD_HID_IO;

	return STATUS_OK;
}

// --- Start a ramp on multiple channels ---
//
// chmask has a 1 bit set for every channel to be started
// mode is composed of the sweep command byte flags SWP_DIRECTION, SWP_CONTINUOUS, SWP_ONCE, and SWP_BIDIR
// the deferred flag is not supported, but reserved for future use. Set it to false when calling this function for compatibility with future library versions
//
LVSTATUS fnLPS_StartRampMC(DEVID deviceID, int mode, int chmask, bool deferred)
{
	int icount;
	unsigned int ch;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	BYTE VNX_ramp[] = { 0, 0, 0, 0, 0, 0 };

	if (mode != 0){
		VNX_ramp[0] = (BYTE)(mode & MODE_SWEEP);	// mode sweep for V2 includes the 0x10 bit
	}
	else {
		VNX_ramp[0] = 0;
	}

	if (TestMode) {
		return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	}

	if (Trace_Out > 2)  printf(" sending a  multi channel ramp command = %x\n", VNX_ramp[0]);

	VNX_ramp[5] = (BYTE)chmask;

	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_ramp, 6)){
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// ---------- profile control functions, only supported in V2 devices -----------

// set profile element, 1 degree units
// profile values are cached, and only read when user code asks for them, writes update the cache

LVSTATUS fnLPS_SetProfileElement(DEVID deviceID, int index, int phase) {
	int phase_20 = phase * 20;	// convert to .05 degree units
	int ch = 0;
	unsigned int bank;
	int old_element;
	int scaled_phase;
	int max_index;
	int cmdlen;
	int tmp;

	if (deviceID >= MAXDEVICES) {
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	max_index = lps[deviceID].ProfileMaxLength - 1;
	if (index < 0 || index > max_index)			// check if the index is valid
	{
		return BAD_PARAMETER;					// the profile index is invalid
	}

	ch = lps[deviceID].Channel;
	if (ch >= CHANNEL_MAX) ch = 0;				// just defending our arrays, should never happen

	bank = (unsigned int)(ch / 8) + 1;		// the logical bank is incremented before it is transmitted
											// this lets us determine if we have a logical bank in the command packet


	if ((phase_20 >= lps[deviceID].MinPhase) && (phase_20 <= lps[deviceID].MaxPhase)) {
		// keep the existing cached value in case our IO fails
		// and stash the new value since we are optimistic

		old_element = lps[deviceID].CachedProfileValue[ch];			// keep cached value in case IO fails

		tmp = index << 16;

		lps[deviceID].CachedProfileValue[ch] =  tmp | (phase_20 & 0xFFFF);	// and stash the new value (high word is index, low word is value)

		if (TestMode) return STATUS_OK;		// update our caches, but don't talk to the real HW (the TestMode version does not simulate the profile)

	}
	else {
		return BAD_PARAMETER;							// the phase value is out of range
	}

	// -- our parameters are good, so lets send them to the hardware

	// -- scale our phase angle value to the units used by this device --
	// As of 6-20-16 we use the device's UnitScale parameter for scaling,
	// and now we are of course scaling from .05 degree units
	if (lps[deviceID].UnitScale > 0)
		scaled_phase = phase_20 / lps[deviceID].UnitScale;

	// -- construct the command --
	BYTE VNX_command[] = {0, 0, 0, 0, 0, 0, 0};

	if (CheckHiRes(deviceID))
	{
		VNX_command[0] = (BYTE)(scaled_phase & 0x00FF);			// Phase low byte
		VNX_command[1] = (BYTE)((scaled_phase >> 8) & 0x00FF);	// Phase high byte
		VNX_command[2] = (BYTE)(index & 0x00FF);				// Index low byte
		VNX_command[3] = (BYTE)((index >> 8) & 0x03);			// Index high bits (only bits 0–1 used)
		VNX_command[4] = (BYTE)bank;							// Logical bank number
		VNX_command[5] = (BYTE)lps[deviceID].ChMask;			// Logical channel mask
	}
	else
	{
		VNX_command[0] = (BYTE)(scaled_phase & 0x00FF);			// Phase low byte
		VNX_command[1] = (BYTE)((scaled_phase >> 8) & 0x00FF);	// Phase high byte
		VNX_command[4] = (BYTE)(index & 0x00FF);
	}

	if (!SendReport(deviceID, VNX_SETPROFILE | VNX_SET, VNX_command, 6)) {
		// our send failed, we'll leave the cache as we found it
		lps[deviceID].CachedProfileValue[ch] = old_element;
		return BAD_HID_IO;
	}
	return STATUS_OK;
}

// Set the time to remain at each phase angle value in a profile

LVSTATUS fnLPS_SetProfileDwellTime(DEVID deviceID, int dwelltime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

	int old_dwelltime = lps[deviceID].ProfileDwellTime[ch];

	if (dwelltime >= VNX_MIN_DWELLTIME) {

		lps[deviceID].ProfileDwellTime[ch] = dwelltime;
		if (TestMode) {
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else {
		return BAD_PARAMETER;
	}

	// the profile dwell time value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileDwellTime[ch];

	if (!SendReport(deviceID, VNX_PROFILEDWELL | VNX_SET, ptr, 4)){

		lps[deviceID].ProfileDwellTime[ch] = old_dwelltime;
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the time to wait at the end of a repeating profile

LVSTATUS fnLPS_SetProfileIdleTime(DEVID deviceID, int idletime) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;
	int old_idletime = lps[deviceID].ProfileIdleTime[ch];

	if (idletime >= VNX_MIN_DWELLTIME){

		lps[deviceID].ProfileIdleTime[ch] = idletime;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else
	{
		return BAD_PARAMETER;
	}

	// the profile idle time value is OK, lets send it to the hardware

	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileIdleTime[ch];


	if (!SendReport(deviceID, VNX_PROFILEIDLE | VNX_SET, ptr, 4)){

		lps[deviceID].ProfileIdleTime[ch] = old_idletime;
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Set the number of elements in the profile
LVSTATUS fnLPS_SetProfileCount(DEVID deviceID, int profilecount) {
	int ch = 0;
	int max_count;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;
	int old_count = lps[deviceID].ProfileCount[ch];

	max_count = lps[deviceID].ProfileMaxLength;
	if (profilecount <= max_count && profilecount > 0){

		lps[deviceID].ProfileCount[ch] = profilecount;
		if (TestMode){
			return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
		}
	}
	else
	{
		return BAD_PARAMETER;
	}

	// the profile count value is OK, lets send it to the hardware
	unsigned char *ptr = (unsigned char *) &lps[deviceID].ProfileCount[ch];

	if (!SendReport(deviceID, VNX_PROFILECOUNT | VNX_SET, ptr, 4)){

		lps[deviceID].ProfileCount[ch] = old_count;
		return BAD_HID_IO;
	}

	return STATUS_OK;
}


// Start the profile immediately
LVSTATUS fnLPS_StartProfile(DEVID deviceID, int mode) {

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	// -- NB: VNX_command[1] and VNX_command[2] must be 0 for immediate sweep start --
	BYTE VNX_command[] = {0, 0, 0, 0};

	if (mode != 0)	// mode is 1 for a single profile, 2 for a repeating profile
	{
		VNX_command[0] = (BYTE) ((mode & 0x00000003) | STATUS_PROFILE_ACTIVE);	// start the profile
	}
	else
	{
		VNX_command[0] = 0;		// stop the profile
	}

	if (TestMode){
		return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	}


	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_command, 3))
	{
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// -- This function uses the multi-channel command capability of V2 firmware devices (also in some earlier devices)
//	  to launch a number of profiles concurrently
//	  chmask has 1 bits for all active channels.
//	  mode is 1 for a single profile, 2 for repeating profiles, and 0 to stop the profiles
//	  the delayed parameter is not supported at this time, set it to false for future compatibility
//
LVSTATUS fnLPS_StartProfileMC(DEVID deviceID, int mode, int chmask, bool delayed)
{
	if (deviceID >= MAXDEVICES) {
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)) {
		return DEVICE_NOT_READY;
	}

	// -- NB: VNX_command[1] and VNX_command[2] must be 0 for immediate sweep start --
	BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0};

	if (mode != 0) {	// mode is 1 for a single profile, 2 for a repeating profile
		VNX_command[0] = (BYTE)((mode & 0x00000003) | STATUS_PROFILE_ACTIVE);	// start the profile
	}
	else {
		VNX_command[0] = 0;		// stop the profile
	}

	if (TestMode) {
		return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
	}

	VNX_command[5] = (BYTE) chmask;

	if (Trace_Out > 2)  printf(" sending a multi channel start profile command = %x\n", VNX_command[0]);

	if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_command, 6)) {
		return BAD_HID_IO;
	}

	return STATUS_OK;
}

// Save the user settings to flash for autonomous operation
LVSTATUS fnLPS_SaveSettings(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  if (TestMode)
    return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW

  char VNX_savesettings[] = {0x42, 0x55, 0x31}; //three byte key to unlock the user protection.

  if (!SendReport(deviceID, VNX_SAVEPAR | VNX_SET, VNX_savesettings, 3))
    return BAD_HID_IO;

  return STATUS_OK;
}

// ------------- Functions to get parameters ---------------------

// Get the working frequency for HiRes Phase Shifters in 100 KHz units
int fnLPS_GetWorkingFrequency(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].WorkingFrequency[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetWorkingFrequency(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].WorkingFrequency[ch];
}

// Get the phase angle setting in 1 degree units
int fnLPS_GetPhaseAngle(DEVID deviceID) {
  int ch = 0;
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;
  
  if (lps[deviceID].PhaseAngle[ch] == -1) {		// we don't have a value cached, so go get it.
    if (!GetPhaseAngle(deviceID)){
      return BAD_HID_IO;
    }
  }

  return lps[deviceID].PhaseAngle[ch] / 20;
}

// Get the ramp start phase angle level
int fnLPS_GetRampStart(DEVID deviceID) {
  int ch = 0;
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].RampStart[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetRampStart(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].RampStart[ch] / 20;
}

// Get the phase angle at the end of the ramp
int fnLPS_GetRampEnd(DEVID deviceID) {
  int ch = 0;
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].RampStop[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetRampStop(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].RampStop[ch] / 20;
}

// Get the time to dwell at each step along the first phase of the ramp
int fnLPS_GetDwellTime(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

    if (lps[deviceID].DwellTime[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetDwellTime(deviceID)){
			return BAD_HID_IO;
		}
	}
    return lps[deviceID].DwellTime[ch];
}

// Get the time to dwell at each step along the second phase of the ramp
int fnLPS_GetDwellTimeTwo(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES)
	  return INVALID_DEVID;

	if (!CheckDeviceOpen(deviceID))
	  return DEVICE_NOT_READY;

	ch = lps[deviceID].Channel;

    if (lps[deviceID].DwellTime2[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetDwellTime2(deviceID)){
			return BAD_HID_IO;
		}
	}
    return lps[deviceID].DwellTime2[ch];
}


// Get the idle time to wait at the end of the ramp
int fnLPS_GetIdleTime(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].IdleTime[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetIdleTime(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].IdleTime[ch];
}

// Get the hold time to wait at the end of the first phase of the ramp
int fnLPS_GetHoldTime(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

  if (lps[deviceID].HoldTime[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetHoldTime(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].HoldTime[ch];
}

// Get the size of the phase angle step for the first phase of the ramp in 1 degree units
int fnLPS_GetPhaseAngleStep(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

    if (lps[deviceID].PhaseStep[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetPhaseAngleStep(deviceID)){
			return BAD_HID_IO;
		}
  }

  return lps[deviceID].PhaseStep[ch] / 20;
}

int fnLPS_GetPhaseAngleStepTwo(DEVID deviceID) {
	int ch = 0;

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  ch = lps[deviceID].Channel;

    if (lps[deviceID].PhaseStep2[ch] == -1) {		// we don't have a value cached, so go get it.
		if (!GetPhaseAngleStep2(deviceID)){
			return BAD_HID_IO;
		}
	}

  return lps[deviceID].PhaseStep2[ch] / 20;
}

// Get functions related to profiles
int fnLPS_GetProfileDwellTime(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;
	if (lps[deviceID].ProfileDwellTime[ch] == -1)	// we don't have a value cached so go get it
		{
			if (!GetProfileDwellTime(deviceID)) {
				return BAD_HID_IO;
			}
		}

	return lps[deviceID].ProfileDwellTime[ch];		// fetch the value for the current channel
}

int fnLPS_GetProfileIdleTime(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

	if (lps[deviceID].ProfileIdleTime[ch] == -1)	// we don't have a value cached
	{
		if (!GetProfileIdleTime(deviceID)) {
				return BAD_HID_IO;
		}
	}

	return lps[deviceID].ProfileIdleTime[ch];	// fetch the value for the current channel
}

int fnLPS_GetProfileCount(DEVID deviceID) {
	int ch = 0;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

		if (lps[deviceID].ProfileCount[ch] == -1) {	// we don't have a value cached, go get one
			if (!GetProfileCount(deviceID)) {
				return BAD_HID_IO;
			}
		}

	return lps[deviceID].ProfileCount[ch];	// fetch the value for the current channel

}


// get profile element, 1 degree units
int fnLPS_GetProfileElement(DEVID deviceID, int index) {
	int ch = 0;
	int max_index;
	int tmp;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	max_index = lps[deviceID].ProfileMaxLength - 1;
	ch = lps[deviceID].Channel;

	if (index < 0 || index > max_index)	{	// check if the index is valid
		return BAD_PARAMETER;
	}
	else {
		// we keep a single profile element cache
		tmp = lps[deviceID].CachedProfileValue[ch] >> 16;		// we keep the index of the cached value in the high 16 bits

		if (tmp == index) return (lps[deviceID].CachedProfileValue[ch] & 0xFFFF) / 20; // we've got a value cached
		else {
			if (!GetProfileElement(deviceID, index)) {
				return BAD_HID_IO;
			}
			return (lps[deviceID].CachedProfileValue[ch] & 0xFFFF) / 20;		// we read the value from the HW and cached it.
		}
	}
}

// -- modified to return DATA_UNAVAILABLE if we cannot get the profile index value in a timely fashion
//		for some profile timing the index is not updated during the profile, or for a long time

int fnLPS_GetProfileIndex(DEVID deviceID){
	int ch = 0;
	int timedout;

	if (deviceID >= MAXDEVICES){
		return INVALID_DEVID;
	}

	if (!CheckDeviceOpen(deviceID)){
		return DEVICE_NOT_READY;
	}

	ch = lps[deviceID].Channel;

	if (lps[deviceID].ProfileIndex[ch] == -1) {
		// we don't have a value cached, we should have one from status reports, but don't, so wait some
		starttime = time(NULL);
		timedout = 0;

		// wait until the device reports its index or 1 second has gone by
		while ((lps[deviceID].ProfileIndex[ch] == -1) && (timedout == 0)) {
	  		catnap(10);	// yield for 10 ms
	  		if ((time(NULL)-starttime) > 1) timedout = 1;
		}

		if (timedout == 1) {
			return DATA_UNAVAILABLE;
		}
	}
	return lps[deviceID].ProfileIndex[ch];
}

// Get the maximum profile length supported by the device
int fnLPS_GetProfileMaxLength(DEVID deviceID)
{
	if (deviceID >= MAXDEVICES)
		return INVALID_DEVID;

	return lps[deviceID].ProfileMaxLength;
}

// Get the maximum phase shift in 1 degree units
int fnLPS_GetMaxPhaseShift(DEVID deviceID) {

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].MaxPhase / 20;
}

// Get the minimum phase shift in 1 degree units
int fnLPS_GetMinPhaseShift(DEVID deviceID) {

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].MinPhase / 20;
}

// Get the resolution of the phase shifter, in 1 degree units
int fnLPS_GetMinPhaseStep(DEVID deviceID) {

  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lps[deviceID].MinPhaseStep / 20;
}

// get the minimum working frequency of the HiRes units
int fnLPS_GetMinWorkingFrequency(DEVID deviceID)
{
	if (deviceID >= MAXDEVICES){ return INVALID_DEVID;}

	return lps[deviceID].MinFrequency;	// even quad devices only have one frequency range for all channels
}

// get the maximum working frequency of the HiRes units
int fnLPS_GetMaxWorkingFrequency(DEVID deviceID)
{
	if (deviceID >= MAXDEVICES){ return INVALID_DEVID;}

	return lps[deviceID].MaxFrequency;		// even quad devices only have one frequency range for all channels
}

// Get a set of bits describing the features available in this phase shifter device
int fnLPS_GetFeatures(DEVID deviceID) {
	int temp = DEFAULT_FEATURES;

	if (deviceID >= MAXDEVICES){ return INVALID_DEVID;}

	else if (CheckHiRes(deviceID)){
		temp = (HAS_HIRES | HAS_BIDIR_RAMPS | HAS_PROFILES);// HiRes has bidirectional ramps, profiles and HiRes
	}
	else{
		temp = (HAS_BIDIR_RAMPS | HAS_PROFILES);			// V2 have bidirectional ramps and profiles
	}

	if (lps[deviceID].NumChannels == 4)    temp |= HAS_4CHANNELS;
  	if (lps[deviceID].NumChannels == 8)    temp |= HAS_8CHANNELS;
	if (lps[deviceID].ProfileMaxLength == 1000) temp |= HAS_LONG_PROFILE;

	return temp;
}
